<?php

/*
 +--------------------------------------------------------------------------+
 | Kolab Sync (ActiveSync for Kolab)                                        |
 |                                                                          |
 | Copyright (C) 2011-2012, Kolab Systems AG                                |
 |                                                                          |
 | This program is free software: you can redistribute it and/or modify     |
 | it under the terms of the GNU Affero General Public License as published |
 | by the Free Software Foundation, either version 3 of the License, or     |
 | (at your option) any later version.                                      |
 |                                                                          |
 | This program is distributed in the hope that it will be useful,          |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of           |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the             |
 | GNU Affero General Public License for more details.                      |
 |                                                                          |
 | You should have received a copy of the GNU Affero General Public License |
 | along with this program. If not, see <http://www.gnu.org/licenses/>      |
 +--------------------------------------------------------------------------+
 | Author: Aleksander Machniak <machniak@kolabsys.com>                      |
 +--------------------------------------------------------------------------+
*/

/**
 * Class for logging messages into log file(s)
 */
class kolab_sync_logger extends Zend_Log
{
    public $mode;

    protected $log_driver;
    protected $logfile;
    protected $format;
    protected $log_dir;
    protected $username;
    protected $log_context;

    /**
     * Constructor
     */
    public function __construct($mode = null)
    {
        $rcube = rcube::get_instance();

        $this->mode    = intval($mode);
        $this->logfile = $rcube->config->get('activesync_log_file');
        $this->log_driver = $rcube->config->get('log_driver');
        $this->format  = $rcube->config->get('log_date_format', 'd-M-Y H:i:s O');
        $this->log_dir = $rcube->config->get('log_dir');

        $r = new ReflectionClass($this);
        $this->_priorities = $r->getConstants();
    }

    public function __call($method, $params)
    {
        $method = strtoupper($method);
        if ($this->_priorities[$method] <= $this->mode) {
            $this->log(array_shift($params), $method);
        }
    }

    /**
     * Check whether debug logging is enabled
     *
     * @return bool
     */
    public function hasDebug()
    {
        // This is what we check in self::log() below
        return (!empty($this->log_dir) || ($this->log_driver == 'logfmt')) && $this->mode >= self::NOTICE;
    }

    /**
     * Message logger
     *
     * @param string     $message Log message
     * @param int|string $method  Message severity
     */
    public function log($message, $method, $extras = null)
    {
        if (is_numeric($method)) {
            $mode   = $method;
            $method = array_search($method, $this->_priorities);
        } else {
            $mode = $this->_priorities[$method];
        }

        // Don't log messages with lower prio than the configured one
        if ($mode > $this->mode) {
            return;
        }

        // write message in logfmt format with extra info when configured to log to STDOUT
        if ($this->log_driver == 'logfmt') {
            $user_name = $this->username;

            switch ($mode) {
                case self::DEBUG:
                    $name = 'debug';
                    break;
                case self::INFO:
                case self::NOTICE:
                    $name = 'console';
                    break;
                default:
                    $name = 'error';
                    break;
            }

            $output = "name=$name component=syncroton";
            if (!empty($this->log_context)) {
                foreach ($this->log_context as $key => $value) {
                    $output .= " $key=$value";
                }
            }

            if (!empty($user_name)) {
                $output .= " user=$user_name";
            }

            if (!is_string($message)) {
                $message = var_export($message, true);
            }

            $line = json_encode($message);
            $output .= " log=$line\n";
            file_put_contents("php://stdout", $output, FILE_APPEND) !== false;
            return;
        }

        // Don't log debug messages if it's disabled e.g. by per_user_logging
        if (empty($this->log_dir) && $mode >= self::NOTICE) {
            return;
        }

        $rcube   = rcube::get_instance();
        $log_dir = $this->log_dir ?: $rcube->config->get('log_dir');
        $logfile = $this->logfile;

        // if log_file is configured all logs will go to it
        // otherwise use separate file for info/debug and warning/error
        $file = "undefined";
        if (!$logfile) {
            switch ($mode) {
                case self::DEBUG:
                case self::INFO:
                case self::NOTICE:
                    $file = 'console';
                    break;
                default:
                    $file = 'errors';
                    break;
            }

            $logfile = $log_dir . DIRECTORY_SEPARATOR . $file;

            if (version_compare(version_parse(RCUBE_VERSION), '1.4.0') >= 0) {
                $logfile .= $rcube->config->get('log_file_ext', '.log');
            }
        } elseif ($logfile[0] != '/') {
            $logfile = $log_dir . DIRECTORY_SEPARATOR . $logfile;
        }

        if (!is_string($message)) {
            $message = var_export($message, true);
        }

        // add user/request information to the log
        if ($mode <= self::WARN) {
            $device = [];
            $params = ['cmd' => 'Cmd', 'device' => 'DeviceId', 'type' => 'DeviceType'];

            if (!empty($this->username)) {
                $device['user'] = $this->username;
            }

            foreach ($params as $key => $val) {
                if (isset($_GET[$val])) {
                    $device[$key] = $_GET[$val];
                }
            }

            if (!empty($device)) {
                $message = @json_encode($device) . ' ' . $message;
            }
        }

        $date    = rcube_utils::date_format($this->format);
        $logline = sprintf("[%s]: [%s] %s\n", $date, $method, $message);

        // write message with file name when configured to log to STDOUT
        if ($this->log_driver == 'stdout') {
            $stdout = "php://stdout";
            file_put_contents($stdout, $logline, FILE_APPEND);
            return;
        }

        if ($fp = @fopen($logfile, 'a')) {
            fwrite($fp, $logline);
            fflush($fp);
            fclose($fp);
            return;
        }

        if ($mode <= self::WARN) {
            // send error to PHPs error handler if write to file didn't succeed
            trigger_error($message, E_USER_WARNING);
        }
    }

    /**
     * Set current user name to add into error log
     */
    public function set_username($username)
    {
        $this->username = $username;
    }

    /**
     * Set log context
     */
    public function set_context($context)
    {
        $this->log_context = $context;
    }

    /**
     * Set log directory
     */
    public function set_log_dir($dir)
    {
        $this->log_dir = $dir;
    }
}
