#!/usr/bin/env php
<?php

/*
 +--------------------------------------------------------------------------+
 | Kolab Sync (ActiveSync for Kolab)                                        |
 |                                                                          |
 | Copyright (C) 2024, Apheleia IT AG <contact@apheleia-it.ch>              |
 |                                                                          |
 | This program is free software: you can redistribute it and/or modify     |
 | it under the terms of the GNU Affero General Public License as published |
 | by the Free Software Foundation, either version 3 of the License, or     |
 | (at your option) any later version.                                      |
 |                                                                          |
 | This program is distributed in the hope that it will be useful,          |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of           |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the             |
 | GNU Affero General Public License for more details.                      |
 |                                                                          |
 | You should have received a copy of the GNU Affero General Public License |
 | along with this program. If not, see <http://www.gnu.org/licenses/>      |
 +--------------------------------------------------------------------------+
 | Author: Christian Mollekopf <mollekopf@apheleia-it.ch>                      |
 +--------------------------------------------------------------------------+
*/
require_once 'shared.php';

$opts = rcube_utils::get_opt([
    'o' => 'owner',
    'd' => 'deviceid',
    't' => 'devicetype', // e.g. WindowsOutlook15 or iPhone
    'p' => 'password',
]);

if (!empty($opts['help'])) {
    print(<<<EOF
        The following options exist:
            --owner: Specify the email address of the device user.
            --dry-run: Only print what would be done, don't execute them.
            --deviceid: Only operate on the specified device id.
            --devicetype: Only operate on the specified device type.
            --password: Required for some operations

        Operations:
            --list: List current subscriptions.
            --reset: Remove all subscriptions.
            --enableall: Enable all subscriptions by looking up backend folders. Requires the user password.
            --enableknown: Enable all subscriptions of folders that have been synchronized previously with syncroton.
            --dropempty: Delete empty subscriptions, triggering a re-import from metadata.
            --reinit: Re-run the initialization for new devices. Requires the user password.

        EOF);
    exit(0);
}


$cli = new SyncrotonCli();

$dryRun = null;
if (!empty($opts['dry-run'])) {
    $dryRun = "[DRY-RUN]" ;
}

if (!empty($opts['dropempty'])) {
    $db = $cli->db;
    if ($dryRun) {
        $query = $db->query("SELECT * FROM `syncroton_subscriptions` WHERE `data` = '[]'");
        while ($record = $db->fetch_assoc($query)) {
            print("$dryRun {$record['device_id']} {$record['type']}\n");
        }
    } else {
        $query = $db->query("DELETE FROM `syncroton_subscriptions` WHERE `data` = '[]'");
        $affectedRows = $db->affected_rows($query);
        print("Deleted $affectedRows empty subscriptions sets.");
    }
    exit(0);
}

if (empty($opts['owner'])) {
    \rcube::raise_error("Owner not specified (--owner).", false, true);
}

$userid = $cli->selectUser($opts['owner'] ?? null);
$devices = $cli->selectDevices($opts['deviceid'] ?? null, $opts['devicetype'] ?? null);

if (empty($devices)) {
    rcube::raise_error("Device not found.", false, true);
}

print("Found devices: " . json_encode($devices) . "\n");

if (!empty($opts['reset'])) {
    $db = $cli->db;
    if ($dryRun) {
        $query = $db->query("SELECT * FROM `syncroton_subscriptions` WHERE `device_id` IN (" . $db->array2list($devices) . ")");
        while ($record = $db->fetch_assoc($query)) {
            print("$dryRun {$record['device_id']} {$record['type']}\n");
        }
    } else {
        $query = $db->query("DELETE FROM `syncroton_subscriptions` WHERE `device_id` IN (" . $db->array2list($devices) . ")");
        $affectedRows = $db->affected_rows($query);
        print("Deleted $affectedRows subscriptions sets.");
    }
}


if (!empty($opts['enableall'])) {
    if (empty($opts['password'])) {
        \rcube::raise_error("Password for owner not specified (--password).", false, true);
    }
    // Init the authentication
    $cli->authenticate($opts['password']);
    $subscriptions = new kolab_subscriptions();
    foreach (['mail', 'event', 'contact', 'task', 'note'] as $type) {
        $data = array_map(function ($v) { return [$v[0], 1]; }, $subscriptions->list_folders($type));
        print_r($data);
        foreach ($devices as $deviceid) {
            $subscriptions->set_subscriptions($deviceid, $type, $data);
        }
    }
}

if (!empty($opts['enableknown'])) {
    $db = $cli->db;
    foreach ($devices as $deviceid) {
        $folderSelect = $db->query(
            "SELECT * FROM `syncroton_folder` WHERE `device_id` = ?",
            $deviceid
        );
        $folders = [
            "Email" => [],
            "Tasks" => [],
            "Calendar" => [],
            "Contacts" => [],
        ];
        while ($folder = $db->fetch_assoc($folderSelect)) {
            //TODO handle subfolders by assembling the folder path
            if ($folder['parentid']) {
                // print("Skipping subfolder: " . $folder['displayname'] . "\n");
                continue;
            }
            // print("Found folder: " . $folder['displayname'] . "\n");
            $displayname = $folder['displayname'];
            $class = $folder['class']; //Email/Tasks/Calendar/Contacts
            $folders[$class][] = $displayname;
        }
        foreach ($folders as $class => $folders) {
            $subscriptions = [];
            foreach ($folders as $folder) {
                $subscriptions[$folder] = 1;
            }
            switch ($class) {
                case "Email":
                    $type = "mail";
                    break;
                case "Calendar":
                    $type = "event";
                    break;
                case "Tasks":
                    $type = "task";
                    break;
                case "Contacts":
                    $type = "contact";
                    break;
                default:
                    print("Unknown class $class\n");
                    continue 2;
            }
            print("$dryRun Setting subscriptions for device $deviceid and type $type: \n" . json_encode($subscriptions) . "\n");
            if (!$dryRun) {
                $cli->setSubscriptions($deviceid, $type, $subscriptions);
            }
        }
    }
}

if (!empty($opts['list'])) {
    foreach ($devices as $deviceid) {
        foreach (['mail', 'event', 'contact', 'task', 'note'] as $type) {
            $subscriptions = $cli->getSubscriptions($deviceid, $type);
            print("Subscriptions for device $deviceid and type $type: \n" . json_encode($subscriptions) . "\n");
        }
    }
}

if (!empty($opts['reinit'])) {
    if (empty($opts['password'])) {
        \rcube::raise_error("Password for owner not specified (--password).", false, true);
    }
    $storage = $cli->getStorage($opts['password']);
    foreach ($devices as $deviceid) {
        print("$dryRun Device init: {$deviceid}\n");
        if (!$dryRun) {
            $storage->device_init($cli->getDeviceId($deviceid));
        }
    }
}
