<?php

class TimezoneConverterTest extends PHPUnit\Framework\TestCase
{
    public function test_list_timezones()
    {
        //        date_default_timezone_set('America/Los_Angeles');

        $converter = kolab_sync_timezone_converter::getInstance();
        $output = $converter->getListOfTimezones('AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAoAAAAEAAIAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMAAAAFAAEAAAAAAAAAxP///w==');

        $this->assertTrue(is_array($output));

        $converter = kolab_sync_timezone_converter::getInstance();
        $output    = $converter->getListOfTimezones('xP///0MAZQBuAHQAcgBhAGwAIABFAHUAcgBvAHAAZQAgAFMAdABhAG4AZABhAHIAZAAgAFQAaQBtAGUAAAAAAAAAAAAAAAoAAAAFAAMAAAAAAAAAAAAAAEMAZQBuAHQAcgBhAGwAIABFAHUAcgBvAHAAZQAgAEQAYQB5AGwAaQBnAGgAdAAgAFQAaQBtAGUAAAAAAAAAAAAAAAMAAAAFAAIAAAAAAAAAxP///w==');

        $this->assertTrue(is_array($output));
        $this->assertTrue(isset($output['Europe/Warsaw']));

        $converter = kolab_sync_timezone_converter::getInstance();
        $output    = $converter->getListOfTimezones('4AEAAFAAYQBjAGkAZgBpAGMAIABTAHQAYQBuAGQAYQByAGQAIABUAGkAbQBlAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAsAAAABAAIAAAAAAAAAAAAAAFAAYQBjAGkAZgBpAGMAIABEAGEAeQBsAGkAZwBoAHQAIABUAGkAbQBlAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMAAAACAAIAAAAAAAAAxP///w==');

        $this->assertTrue(is_array($output));
        $this->assertTrue(isset($output['America/Los_Angeles']));

        $converter = kolab_sync_timezone_converter::getInstance();
        $output = $converter->getListOfTimezones('Lv///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA==');

        $this->assertTrue(is_array($output));
        $this->assertTrue(isset($output['Asia/Tehran']));

        // As seen in outlook
        $converter = kolab_sync_timezone_converter::getInstance();
        $output = $converter->getListOfTimezones('Lv///0kAcgBhAG4AIABTAHQAYQBuAGQAYQByAGQAIABUAGkAbQBlAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAkABAADABcAOwA7AOcDAAAAAEkAcgBhAG4AIABEAGEAeQBsAGkAZwBoAHQAIABUAGkAbQBlAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMAAwAEAAAAAAAAAAAAxP///w==');

        $this->assertTrue(is_array($output));
        $this->assertTrue(isset($output['Asia/Tehran']));
    }

    public static function data_get_timezone()
    {
        return [
            ['UTC'],
            ['Europe/Warsaw'],
            ['Europe/Zurich'],
            ['America/Los_Angeles'],
            ['Asia/Tehran'],
        ];
    }

    /**
     * @dataProvider data_get_timezone
     */
    public function test_get_timezone($tzName)
    {
        date_default_timezone_set('America/Los_Angeles');

        $converter = kolab_sync_timezone_converter::getInstance();
        $datetime  = '2017-01-01T12:00:00Z';

        $offsets = $converter->getOffsetsForTimezone($tzName, $datetime);
        $output  = $converter->getTimezone($offsets, $tzName);
        $this->assertSame($tzName, $output);
    }

    public function test_get_offsets_for_timezone()
    {
        date_default_timezone_set('America/Los_Angeles');

        $converter = kolab_sync_timezone_converter::getInstance();
        $datetime  = '2017-01-01T12:00:00Z';

        $output = $converter->getOffsetsForTimezone('UTC', $datetime);

        $this->assertSame($output['bias'], 0);
        $this->assertSame($output['standardBias'], 0);
        $this->assertSame($output['daylightBias'], 0);
        $this->assertSame($output['standardMonth'], 0);
        $this->assertSame($output['daylightMonth'], 0);

        $output = $converter->getOffsetsForTimezone('Europe/Warsaw', $datetime);

        $this->assertSame($output['standardBias'], 0);
        $this->assertSame($output['standardMonth'], 10);
        $this->assertSame($output['standardWeek'], 5);
        $this->assertSame($output['standardHour'], 3);
        $this->assertSame($output['daylightBias'], -60);
        $this->assertSame($output['daylightMonth'], 3);
        $this->assertSame($output['daylightWeek'], 5);
        $this->assertSame($output['daylightHour'], 2);

        $output = $converter->getOffsetsForTimezone('America/Los_Angeles', $datetime);

        $this->assertSame($output['bias'], 480);
        $this->assertSame($output['standardBias'], 0);
        $this->assertSame($output['standardMonth'], 11);
        $this->assertSame($output['standardWeek'], 1);
        $this->assertSame($output['standardHour'], 2);
        $this->assertSame($output['daylightBias'], -60);
        $this->assertSame($output['daylightMonth'], 3);
        $this->assertSame($output['daylightWeek'], 2);
        $this->assertSame($output['daylightHour'], 2);

        $output = $converter->getOffsetsForTimezone('Atlantic/Azores', $datetime);

        $this->assertSame($output['bias'], 60);
        $this->assertSame($output['standardBias'], 0);
        $this->assertSame($output['standardMonth'], 10);
        $this->assertSame($output['standardWeek'], 5);
        $this->assertSame($output['standardHour'], 1);
        $this->assertSame($output['daylightBias'], -60);
        $this->assertSame($output['daylightMonth'], 3);
        $this->assertSame($output['daylightWeek'], 5);
        $this->assertSame($output['daylightHour'], 0);

        //Check before dst change
        $output = $converter->getOffsetsForTimezone('Asia/Tehran', $datetime);

        $this->assertSame($output['bias'], -210);
        $this->assertSame($output['standardBias'], 0);
        $this->assertSame($output['standardMonth'], 9);
        $this->assertSame($output['standardWeek'], 3);
        $this->assertSame($output['standardDayOfWeek'], 4);
        $this->assertSame($output['standardHour'], 24);
        $this->assertSame($output['daylightBias'], -60);
        $this->assertSame($output['daylightMonth'], 3);
        $this->assertSame($output['daylightWeek'], 4);
        $this->assertSame($output['daylightDayOfWeek'], 3);
        $this->assertSame($output['daylightHour'], 0);

        //Check after dst change
        $output = $converter->getOffsetsForTimezone('Asia/Tehran', '2023-01-01T12:00:00Z');

        $this->assertSame($output['bias'], -210);
        $this->assertSame($output['standardBias'], 0);
        $this->assertSame($output['standardMonth'], 0);
        $this->assertSame($output['standardWeek'], 0);
        $this->assertSame($output['standardDayOfWeek'], 0);
        $this->assertSame($output['standardHour'], 0);
        $this->assertSame($output['daylightBias'], 0);
        $this->assertSame($output['daylightMonth'], 0);
        $this->assertSame($output['daylightWeek'], 0);
        $this->assertSame($output['daylightDayOfWeek'], 0);
        $this->assertSame($output['daylightHour'], 0);
    }

    public static function data_timezone_conversion()
    {
        return [
            //Pre dst change
            ['Asia/Tehran', 'Lv///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAkAAgADABcAOwA7AOcDAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMAAQAEAAAAAAAAAAAAxP///w==', '2021-07-01T12:00:00Z'],
            //Post dst change
            ['Asia/Tehran', 'Lv///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA==', '2023-04-01T12:00:00Z'],
            ['Pacific/Pago_Pago', 'lAIAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA==', '2021-07-01T12:00:00Z'],
            ['Europe/Warsaw', 'xP///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAoAAAAFAAMAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMAAAAFAAIAAAAAAAAAxP///w==', '2021-07-01T12:00:00Z'],
        ];
    }

    /**
     * @dataProvider data_timezone_conversion
     */
    public function test_timezone_conversion($tz, $expected, $datetime)
    {
        $converter = kolab_sync_timezone_converter::getInstance();
        $output = $converter->encodeTimezone($tz, $datetime);

        $this->assertSame($expected, $output);

        $output = $converter->getListOfTimezones($output);

        $this->assertTrue(is_array($output));
        $this->assertTrue(isset($output[$tz]));
    }
}
