<?php

namespace Tests\Sync\Sync;

class RelationsTest extends \Tests\SyncTestCase
{
    protected function initialSyncRequest($folderId)
    {
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync">
                <Collections>
                    <Collection>
                        <SyncKey>0</SyncKey>
                        <CollectionId>{$folderId}</CollectionId>
                    </Collection>
                </Collections>
            </Sync>
            EOF;
        return $this->request($request, 'Sync');
    }

    protected function syncRequest($syncKey, $folderId, $windowSize = null)
    {
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase">
                <Collections>
                    <Collection>
                        <SyncKey>{$syncKey}</SyncKey>
                        <CollectionId>{$folderId}</CollectionId>
                        <DeletesAsMoves>1</DeletesAsMoves>
                        <GetChanges>1</GetChanges>
                        <WindowSize>{$windowSize}</WindowSize>
                        <Options>
                            <FilterType>0</FilterType>
                            <Conflict>1</Conflict>
                            <BodyPreference xmlns="uri:AirSyncBase">
                                <Type>2</Type>
                                <TruncationSize>51200</TruncationSize>
                                <AllOrNone>0</AllOrNone>
                            </BodyPreference>
                        </Options>
                    </Collection>
                </Collections>
            </Sync>
            EOF;
        return $this->request($request, 'Sync');
    }

    private function getRelationsState($device_id, $folderId)
    {
        $db    = \rcube::get_instance()->get_dbh();
        $result = $db->query(
            "SELECT `data`, `synctime` FROM `syncroton_relations_state`"
            . " WHERE `device_id` IN (SELECT `id` FROM `syncroton_device` WHERE `deviceid` = ?) AND `folder_id` = ?"
            . " ORDER BY `synctime` DESC",
            $device_id,
            $folderId
        );
        $data = [];
        while ($state = $db->fetch_assoc($result)) {
            $data[] = $state;
        }
        return $data;
    }

    /**
     * Test Sync command
     */
    public function testRelationsSync()
    {
        $sync = \kolab_sync::get_instance();

        $this->emptyTestFolder('INBOX', 'mail');
        if ($this->isStorageDriver('kolab4')) {
            $sync->get_storage()->set_metadata(\kolab_storage_tags::METADATA_ROOT, [\kolab_storage_tags::METADATA_TAGS_KEY => null]);
        } else {
            $this->emptyTestFolder('Configuration', 'configuration');
        }
        $this->registerDevice();

        // Test INBOX
        $folderId = '38b950ebd62cd9a66929c89615d0fc04';
        $syncKey = 0;
        $response = $this->initialSyncRequest($folderId);
        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:Sync/ns:Collections/ns:Collection/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("//ns:Sync/ns:Collections/ns:Collection/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame('Email', $xpath->query("//ns:Sync/ns:Collections/ns:Collection/ns:Class")->item(0)->nodeValue);
        $this->assertSame($folderId, $xpath->query("//ns:Sync/ns:Collections/ns:Collection/ns:CollectionId")->item(0)->nodeValue);

        // First we append
        $uid1 = $this->appendMail('INBOX', 'mail.sync1');
        $uid2 = $this->appendMail('INBOX', 'mail.sync2');
        $this->appendMail('INBOX', 'mail.sync1', ['sync1' => 'sync3']);
        $this->appendMail('INBOX', 'mail.sync1', ['sync1' => 'sync4']);

        $sync = \kolab_sync::get_instance();

        // Add a tag
        $sync->storage(true)->updateItem($folderId, self::$deviceId, \kolab_sync_storage::MODEL_EMAIL, $uid1, null, ['categories' => ['test1']]);
        sleep(1);

        $response = $this->syncRequest($syncKey, $folderId, 10);
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $root = "//ns:Sync/ns:Collections/ns:Collection";
        $this->assertSame('1', $xpath->query("{$root}/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("{$root}/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame($folderId, $xpath->query("{$root}/ns:CollectionId")->item(0)->nodeValue);
        $this->assertSame(4, $xpath->query("{$root}/ns:Commands/ns:Add")->count());

        $root .= "/ns:Commands/ns:Add";
        $this->assertSame(1, $xpath->query("{$root}/ns:ApplicationData/Email:Categories")->count());
        $this->assertSame("test1", $xpath->query("{$root}/ns:ApplicationData/Email:Categories")->item(0)->nodeValue);

        // Add a second tag
        $sync->storage(true)->updateItem($folderId, self::$deviceId, \kolab_sync_storage::MODEL_EMAIL, $uid1, null, ['categories' => ['test1', 'test2']]);
        sleep(1); // Necessary to make sure we pick up on the tag.

        $response = $this->syncRequest($syncKey, $folderId, 10);
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = "//ns:Sync/ns:Collections/ns:Collection";
        $this->assertSame('1', $xpath->query("{$root}/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("{$root}/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame($folderId, $xpath->query("{$root}/ns:CollectionId")->item(0)->nodeValue);
        $this->assertSame(0, $xpath->query("{$root}/ns:Commands/ns:Add")->count());
        $this->assertSame(1, $xpath->query("{$root}/ns:Commands/ns:Change")->count());
        $root .= "/ns:Commands/ns:Change";
        $this->assertSame(1, $xpath->query("{$root}/ns:ApplicationData/Email:Categories")->count());
        //FIXME not sure what I'm doing wrong, but the xml looks ok
        $this->assertSame("test1test2", $xpath->query("{$root}/ns:ApplicationData/Email:Categories")->item(0)->nodeValue);

        $retries = 2;
        $syncKey--;
        // Rerun the same command and make sure we get the same result
        for ($i = 0; $i < $retries; $i++) {
            $response = $this->syncRequest($syncKey, $folderId, 10);
            $this->assertEquals(200, $response->getStatusCode());
            $dom = $this->fromWbxml($response->getBody());
            $xpath = $this->xpath($dom);

            $root = "//ns:Sync/ns:Collections/ns:Collection";
            $this->assertSame('1', $xpath->query("{$root}/ns:Status")->item(0)->nodeValue);
            $this->assertSame($folderId, $xpath->query("{$root}/ns:CollectionId")->item(0)->nodeValue);
            $this->assertSame(0, $xpath->query("{$root}/ns:Commands/ns:Add")->count());
            $this->assertSame(1, $xpath->query("{$root}/ns:Commands/ns:Change")->count());
            $root .= "/ns:Commands/ns:Change";
            $this->assertSame(1, $xpath->query("{$root}/ns:ApplicationData/Email:Categories")->count());
            //FIXME not sure what I'm doing wrong, but the xml looks ok
            $this->assertSame("test1test2", $xpath->query("{$root}/ns:ApplicationData/Email:Categories")->item(0)->nodeValue);

            // Assert the db state
            if ($this->isStorageDriver('kolab')) {
                $this->assertSame(2, count($this->getRelationsState(self::$deviceId, $folderId)));
            }
            // Make sure we have a new timestamp after the first iteration.
            // This way we can potentially catch errors when we end up using the same or a different timestamp.
            sleep(1);
        }
        $syncKey += ($retries + 1);

        // Reset to no tags
        $sync->storage(true)->updateItem($folderId, self::$deviceId, \kolab_sync_storage::MODEL_EMAIL, $uid1, null, ['categories' => []]);
        sleep(1); // Necessary to make sure we pick up on the tag.

        $response = $this->syncRequest($syncKey, $folderId, 10);
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = "//ns:Sync/ns:Collections/ns:Collection";
        $this->assertSame('1', $xpath->query("{$root}/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("{$root}/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame($folderId, $xpath->query("{$root}/ns:CollectionId")->item(0)->nodeValue);
        $this->assertSame(0, $xpath->query("{$root}/ns:Commands/ns:Add")->count());
        $this->assertSame(1, $xpath->query("{$root}/ns:Commands/ns:Change")->count());
        $root .= "/ns:Commands/ns:Change";
        $this->assertSame(0, $xpath->query("{$root}/ns:ApplicationData/Email:Categories")->count());
        //FIXME this currently fails because we omit the empty categories element
        // $this->assertSame("", $xpath->query("{$root}/ns:ApplicationData/Email:Categories")->item(0)->nodeValue);

        // Assert the db state
        if ($this->isStorageDriver('kolab')) {
            $this->assertSame(2, count($this->getRelationsState(self::$deviceId, $folderId)));
        }

        $response = $this->syncRequest($syncKey, $folderId, 10);
        $this->assertEquals(200, $response->getStatusCode());
        // We expect an empty response without a change
        $this->assertEquals(0, $response->getBody()->getSize());

        return $syncKey;
    }

    public function testPing()
    {
        // Setup with a tag and an initial sync completed
        $folderId = '38b950ebd62cd9a66929c89615d0fc04';
        $sync = \kolab_sync::get_instance();

        $uid1 = $this->appendMail('INBOX', 'mail.sync1');
        $sync->storage(true)->updateItem($folderId, self::$deviceId, \kolab_sync_storage::MODEL_EMAIL, $uid1, null, ['categories' => ['test1']]);
        sleep(1);

        $response = $this->syncRequest(0, $folderId, 10);
        $this->assertEquals(200, $response->getStatusCode());
        $response = $this->syncRequest(1, $folderId, 10);
        $this->assertEquals(200, $response->getStatusCode());

        if ($this->isStorageDriver('kolab')) {
            $this->assertSame(2, count($this->getRelationsState(self::$deviceId, $folderId)));
        }

        // Make sure the timestamp changes
        sleep(1);

        // Pings should not change the number of relation states
        for ($i = 0; $i < 2; $i++) {
            $request = <<<EOF
                <?xml version="1.0" encoding="utf-8"?>
                <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
                <Ping xmlns="uri:Ping">
                    <HeartbeatInterval>0</HeartbeatInterval>
                    <Folders>
                        <Folder>
                            <Id>$folderId</Id>
                            <Class>Email</Class>
                        </Folder>
                    </Folders>
                </Ping>
                EOF;

            $response = $this->request($request, 'Ping');

            $this->assertEquals(200, $response->getStatusCode());
            if ($this->isStorageDriver('kolab')) {
                $this->assertSame(1, count($this->getRelationsState(self::$deviceId, $folderId)));
            }
        }

        // This simulates a specific case where we had:
        // * An old lastsync timestamp (because the folders were not actively synchronized)
        // * The folder was still included in the ping command
        // => This resulted in the relations code never finding a relation, and thus not cleaning up, but it still inserted new entries
        $db    = \rcube::get_instance()->get_dbh();
        $result = $db->query(
            "UPDATE `syncroton_synckey` SET `lastsync` = '2023-06-23 13:15:03'"
        );

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Ping xmlns="uri:Ping">
                <HeartbeatInterval>0</HeartbeatInterval>
                <Folders>
                    <Folder>
                        <Id>$folderId</Id>
                        <Class>Email</Class>
                    </Folder>
                </Folders>
            </Ping>
            EOF;

        $response = $this->request($request, 'Ping');

        $this->assertEquals(200, $response->getStatusCode());
        if ($this->isStorageDriver('kolab')) {
            $this->assertSame(1, count($this->getRelationsState(self::$deviceId, $folderId)));
        }
    }
}
