<?php

/*
 +--------------------------------------------------------------------------+
 | Kolab Sync (ActiveSync for Kolab)                                        |
 |                                                                          |
 | Copyright (C) 2011-2012, Kolab Systems AG <contact@kolabsys.com>         |
 |                                                                          |
 | This program is free software: you can redistribute it and/or modify     |
 | it under the terms of the GNU Affero General Public License as published |
 | by the Free Software Foundation, either version 3 of the License, or     |
 | (at your option) any later version.                                      |
 |                                                                          |
 | This program is distributed in the hope that it will be useful,          |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of           |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the             |
 | GNU Affero General Public License for more details.                      |
 |                                                                          |
 | You should have received a copy of the GNU Affero General Public License |
 | along with this program. If not, see <http://www.gnu.org/licenses/>      |
 +--------------------------------------------------------------------------+
 | Author: Aleksander Machniak <machniak@kolabsys.com>                      |
 +--------------------------------------------------------------------------+
*/

/**
 * Kolab backend class for the folder state storage
 */
class kolab_sync_backend_folder extends kolab_sync_backend_common implements Syncroton_Backend_IFolder
{
    protected $table_name     = 'syncroton_folder';
    protected $interface_name = 'Syncroton_Model_IFolder';

    /**
     * Delete all stored folder ids for a given device
     *
     * @param Syncroton_Model_Device|string $deviceid Device object or identifier
     */
    public function resetState($deviceid)
    {
        $device_id = $deviceid instanceof Syncroton_Model_IDevice ? $deviceid->id : $deviceid;

        $where[] = $this->db->quote_identifier('device_id') . ' = ' . $this->db->quote($device_id);

        $this->db->query('DELETE FROM `' . $this->table_name . '` WHERE ' . implode(' AND ', $where));
    }

    /**
     * mark folder as deleted. The state gets removed finally,
     * when the synckey gets validated during next sync.
     *
     * @param Syncroton_Model_IFolder|string $id
     */
    public function delete($id)
    {
        $id = $id instanceof Syncroton_Model_IFolder ? $id->id : $id;

        $result = $this->db->query("UPDATE `{$this->table_name}` SET `is_deleted` = 1 WHERE `id` = ?", [$id]);

        return $result;
    }

    /**
     * Get array of ids which got send to the client for a given class
     *
     * @param Syncroton_Model_Device|string $deviceid Device object or identifier
     * @param string                        $class    Class name
     * @param int                           $syncKey  Sync key
     *
     * @return array List of object identifiers
     */
    public function getFolderState($deviceid, $class, $syncKey = null)
    {
        $device_id = $deviceid instanceof Syncroton_Model_IDevice ? $deviceid->id : $deviceid;

        $where[] = $this->db->quote_identifier('device_id') . ' = ' . $this->db->quote($device_id);
        $where[] = $this->db->quote_identifier('class') . ' = ' . $this->db->quote($class);
        if ($syncKey) {
            $where[] = $this->db->quote_identifier('creation_synckey') . ' < ' . $this->db->quote($syncKey + 1);
        }

        $select = $this->db->query('SELECT * FROM `' . $this->table_name . '` WHERE ' . implode(' AND ', $where));
        $result = [];

        while ($folder = $this->db->fetch_assoc($select)) {
            $result[$folder['folderid']] = $this->get_object($folder);
        }

        return $result;
    }

    /**
     * Get folder
     *
     * @param Syncroton_Model_Device|string  $deviceid Device object or identifier
     * @param string                         $folderid Folder identifier
     *
     * @return Syncroton_Model_IFolder Folder object
     */
    public function getFolder($deviceid, $folderid)
    {
        $device_id = $deviceid instanceof Syncroton_Model_IDevice ? $deviceid->id : $deviceid;

        $where[] = $this->db->quote_identifier('device_id') . ' = ' . $this->db->quote($device_id);
        $where[] = $this->db->quote_identifier('folderid') . ' = ' . $this->db->quote($folderid);

        $select = $this->db->query('SELECT * FROM `' . $this->table_name . '` WHERE ' . implode(' AND ', $where));
        $folder = $this->db->fetch_assoc($select);

        if (!empty($folder['resync'])) {
            throw new Syncroton_Exception_NotFound("Folder $folderid not found because of resync");
        }
        if (empty($folder)) {
            throw new Syncroton_Exception_NotFound("Folder $folderid not found");
        }

        return $this->get_object($folder);
    }

    /**
     * Check if the folder already exists
     *
     * @param Syncroton_Model_Device|string  $deviceid Device object or identifier
     * @param string                         $folderid Folder identifier
     *
     * @return bool true if it exists
     */
    public function exists($deviceid, $folderid)
    {
        $device_id = $deviceid instanceof Syncroton_Model_IDevice ? $deviceid->id : $deviceid;

        $where[] = $this->db->quote_identifier('device_id') . ' = ' . $this->db->quote($device_id);
        $where[] = $this->db->quote_identifier('folderid') . ' = ' . $this->db->quote($folderid);

        $select = $this->db->query('SELECT 1 FROM `' . $this->table_name . '` WHERE ' . implode(' AND ', $where));
        $folder = $this->db->fetch_assoc($select);
        return !empty($folder);
    }

    /**
     * Find out if the folder hierarchy changed since the last FolderSync
     *
     * @param Syncroton_Model_Device $device Device object
     *
     * @return bool True if folders hierarchy changed, False otherwise
     */
    public function hasHierarchyChanges($device)
    {
        $timestamp      = new DateTime('now', new DateTimeZone('utc'));
        $client_crc     = '';
        $server_crc     = '';
        $client_folders = [];
        $server_folders = [];
        $folder_classes = [
            Syncroton_Data_Factory::CLASS_CALENDAR,
            Syncroton_Data_Factory::CLASS_CONTACTS,
            Syncroton_Data_Factory::CLASS_EMAIL,
            Syncroton_Data_Factory::CLASS_NOTES,
            Syncroton_Data_Factory::CLASS_TASKS,
        ];

        // Reset imap cache, metadata cache and the folder list cache so we work with up-to-date folders lists
        rcube::get_instance()->get_storage()->clear_cache('mailboxes', true);
        kolab_sync::storage()->reset();
        foreach ($folder_classes as $class) {
            // @phpstan-ignore-next-line
            Syncroton_Data_Factory::factory($class, $device, $timestamp)->clearCache();
        }

        // Retrieve all folders already sent to the client
        $select = $this->db->query("SELECT * FROM `{$this->table_name}` WHERE `device_id` = ?", $device->id);

        while ($folder = $this->db->fetch_assoc($select)) {
            if (!empty($folder['resync'])) {
                // Folder re-sync requested
                return true;
            }

            $client_folders[$folder['folderid']] = $this->get_object($folder);
        }

        foreach ($folder_classes as $class) {
            try {
                // retrieve all folders available in data backend
                $dataController = Syncroton_Data_Factory::factory($class, $device, $timestamp);
                $server_folders = array_merge($server_folders, $dataController->getAllFolders());
            } catch (Exception $e) {
                rcube::raise_error($e, true, false);
                // This is server error, returning True might cause infinite sync loops
                return false;
            }
        }

        ksort($client_folders);
        ksort($server_folders);

        foreach ($client_folders as $folder) {
            $client_crc .= '^' . $folder->serverId . ':' . $folder->displayName . ':' . $folder->parentId;
        }

        foreach ($server_folders as $folder) {
            $server_crc .= '^' . $folder->serverId . ':' . $folder->displayName . ':' . $folder->parentId;
        }

        if ($client_crc !== $server_crc) {
            if ($_logger = Syncroton_Registry::get('loggerBackend')) {
                foreach ($client_folders as $folder) {
                    if (strpos($server_crc, "^{$folder->serverId}:") === false) {
                        $_logger->debug(__METHOD__ . '::' . __LINE__ . " Folder on the client, but not on the server: {$folder->serverId} {$folder->displayName}");
                    }
                }
                foreach ($server_folders as $folder) {
                    if (strpos($client_crc, "^{$folder->serverId}:") === false) {
                        $_logger->debug(__METHOD__ . '::' . __LINE__ . " Folder on the server, but not on the client: {$folder->serverId} {$folder->displayName}");
                    }
                }
            }
            return true;
        }
        return false;
    }

    /**
     * (non-PHPdoc)
     * @see kolab_sync_backend_common::from_camelcase()
     */
    protected function from_camelcase($string)
    {
        switch ($string) {
            case 'displayName':
            case 'parentId':
                return strtolower($string);
            case 'serverId':
                return 'folderid';
            default:
                return parent::from_camelcase($string);
        }
    }

    /**
     * (non-PHPdoc)
     * @see kolab_sync_backend_common::to_camelcase()
     */
    protected function to_camelcase($string, $ucFirst = true)
    {
        switch ($string) {
            case 'displayname':
                return 'displayName';
            case 'parentid':
                return 'parentId';
            case 'folderid':
                return 'serverId';
            default:
                return parent::to_camelcase($string, $ucFirst);
        }
    }
}
