<?php

/**
 * Kolab files storage
 *
 * @version @package_version@
 * @author Aleksander Machniak <machniak@kolabsys.com>
 *
 * Copyright (C) 2013, Kolab Systems AG <contact@kolabsys.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

class kolab_files extends rcube_plugin
{
    // all task excluding 'login' and 'logout'
    public $task = '?(?!login|logout).*';

    public $allowed_prefs = ['kolab_files_collapsed_folders'];
    public $rc;
    public $home;
    private $engine;

    public function init()
    {
        $this->rc = rcube::get_instance();

        // we use libkolab::http_request() from libkolab with its configuration
        $this->require_plugin('libkolab');

        // Register hooks related to token refresh (must be before is_token_valid() below)
        $this->add_hook('refresh', [$this, 'refresh']);
        $this->add_hook('oauth_refresh_token', [$this, 'oauth_refresh_token']);

        if (isset($this->rc->oauth)) {
            $this->rc->oauth->is_token_valid();
        }

        // Plugin actions for other tasks
        $this->register_action('plugin.kolab_files', [$this, 'actions']);

        // Register task
        $this->register_task('files');

        // Register plugin task actions
        $this->register_action('index', [$this, 'actions']);
        $this->register_action('prefs', [$this, 'actions']);
        $this->register_action('open', [$this, 'actions']);
        $this->register_action('edit', [$this, 'actions']);
        $this->register_action('share', [$this, 'actions']);
        $this->register_action('reset', [$this, 'actions']);
        $this->register_action('autocomplete', [$this, 'autocomplete']);

        // Load UI from startup hook
        $this->add_hook('startup', [$this, 'startup']);

        $this->add_hook('preferences_save', [$this, 'preferences_save']);
    }

    /**
     * Creates kolab_files_engine instance
     */
    private function engine()
    {
        if ($this->engine === null) {
            // the files module can be enabled/disabled by the kolab_auth plugin
            if ($this->rc->config->get('kolab_files_disabled') || !$this->rc->config->get('kolab_files_enabled', true)) {
                return $this->engine = false;
            }

            $this->load_config();

            $client_url = $this->rc->config->get('kolab_files_url');
            $server_url = $this->rc->config->get('kolab_files_server_url');

            if (!$client_url) {
                return $this->engine = false;
            }

            require_once $this->home . '/lib/kolab_files_engine.php';

            $this->engine = new kolab_files_engine($this, $client_url, $server_url);
        }

        return $this->engine;
    }

    /**
     * Startup hook handler, initializes/enables Files UI
     */
    public function startup($args)
    {
        // call this from startup to give a chance to set
        // kolab_files_enabled/disabled in kolab_auth plugin
        $this->ui();
    }

    /**
     * Adds elements of files API user interface
     */
    private function ui()
    {
        if ($this->rc->output->type != 'html') {
            return;
        }

        if ($engine = $this->engine()) {
            $engine->ui();
        }
    }

    /**
     * Refresh hook handler
     */
    public function refresh($args)
    {
        // Here we are refreshing API session, so when we need it
        // the session will be active
        if ($engine = $this->engine()) {
            $this->rc->output->set_env('files_token', $engine->get_api_token());
        }

        return $args;
    }

    /**
     * oauth_refresh_token hook handler
     */
    public function oauth_refresh_token($args)
    {
        // Force a new chwala session, which will update the credentials (which hold the oauth token)
        $_SESSION['kolab_files_token'] = null;
        if ($engine = $this->engine()) {
            $this->rc->output->set_env('files_token', $engine->get_api_token());
        }
    }

    /**
     * Engine actions handler
     */
    public function actions()
    {
        if ($engine = $this->engine()) {
            $engine->actions();
        }
    }

    /**
     * Handler for user autocomplete request
     */
    public function autocomplete()
    {
        $this->load_config();

        require_once $this->home . '/lib/kolab_files_autocomplete.php';

        new kolab_files_autocomplete();
    }

    /**
     * Update chwala config on user preferences change
     */
    public function preferences_save($props)
    {
        if ($props['section'] == 'general') {
            $dont_override = (array) $this->rc->config->get('dont_override');
            $prefs         = [];

            foreach (['timezone', 'date_long'] as $idx) {
                if (isset($props['prefs'][$idx]) && !in_array($idx, $dont_override)) {
                    $prefs[$idx] = $props['prefs'][$idx];
                }
            }

            if (!empty($prefs) && ($engine = $this->engine())) {
                $engine->configure(false, $prefs);
            }
        }
    }
}
