<?php

/*
 +--------------------------------------------------------------------------+
 | Kolab Sync (ActiveSync for Kolab)                                        |
 |                                                                          |
 | Copyright (C) 2011-2023, Apheleia IT AG <contact@apheleia-it.ch>         |
 |                                                                          |
 | This program is free software: you can redistribute it and/or modify     |
 | it under the terms of the GNU Affero General Public License as published |
 | by the Free Software Foundation, either version 3 of the License, or     |
 | (at your option) any later version.                                      |
 |                                                                          |
 | This program is distributed in the hope that it will be useful,          |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of           |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the             |
 | GNU Affero General Public License for more details.                      |
 |                                                                          |
 | You should have received a copy of the GNU Affero General Public License |
 | along with this program. If not, see <http://www.gnu.org/licenses/>      |
 +--------------------------------------------------------------------------+
 | Author: Aleksander Machniak <machniak@kolabsys.com>                      |
 +--------------------------------------------------------------------------+
*/

/**
 * Storage handling class with Kolab 4 support (IMAP + CalDAV + CardDAV)
 */
class kolab_sync_storage_kolab4 extends kolab_sync_storage
{
    protected $davStorage = null;
    protected $tagStorage = null;
    protected $relationSupport = [self::MODEL_EMAIL];

    /**
     * This implements the 'singleton' design pattern
     *
     * @return kolab_sync_storage_kolab4 The one and only instance
     */
    public static function get_instance()
    {
        if (!self::$instance) {
            self::$instance = new kolab_sync_storage_kolab4();
            self::$instance->startup();  // init AFTER object was linked with self::$instance
        }

        return self::$instance;
    }

    /**
     * Class initialization
     */
    public function startup()
    {
        $sync = kolab_sync::get_instance();

        if ($sync->username === null || $sync->password === null) {
            throw new Exception("Unsupported storage handler use!");
        }

        $url = $sync->config->get('activesync_dav_server', 'http://localhost');

        if (strpos($url, '://') === false) {
            $url = 'http://' . $url;
        }

        // Inject user+password to the URL, there's no other way to pass it to the DAV client
        $url = str_replace('://', '://' . rawurlencode($sync->username) . ':' . rawurlencode($sync->password) . '@', $url);

        $this->tagStorage = new kolab_storage_tags();
        $this->davStorage = new kolab_storage_dav($url); // DAV
        $this->storage = $sync->get_storage(); // IMAP

        // set additional header used by libkolab
        $this->storage->set_options([
                'skip_deleted'  => true,
                'threading'     => false,
        ]);

        // Disable paging
        $this->storage->set_pagesize(999999);

        // Folders subscriptions engine
        $this->subscriptions = new kolab_subscriptions($url);
    }

    /**
     * Get list of folders available for sync
     *
     * @param string $deviceid  Device identifier
     * @param string $type      Folder (class) type
     * @param bool   $flat_mode Enables flat-list mode
     *
     * @return array|bool List of mailbox folders, False on backend failure
     */
    public function folders_list($deviceid, $type, $flat_mode = false)
    {
        $list = [];

        // get mail folders subscribed for sync
        if ($type === self::MODEL_EMAIL) {
            $special_folders = $this->storage->get_special_folders(true);
            $type_map = [
                'drafts' => 3,
                'trash' => 4,
                'sent' => 5,
            ];

            // Get the folders "subscribed" for activesync
            foreach ($this->subscriptions->list_subscriptions($deviceid, self::MODEL_EMAIL) as $folder => $meta) {
                // Force numeric folder name to be a string (T1283)
                $folder = (string) $folder;

                // Activesync folder properties
                $folder_data = $this->folder_data($folder, 'mail');

                // Set proper type for special folders
                if (($type = array_search($folder, $special_folders)) && isset($type_map[$type])) {
                    $folder_data['type'] = $type_map[$type];
                }

                $list[$folder_data['serverId']] = $folder_data;
            }
            if ($flat_mode) {
                $list = $this->folders_list_flat_mail($list);
            }
        } elseif (in_array($type, [self::MODEL_CONTACTS, self::MODEL_CALENDAR, self::MODEL_TASKS])) {
            if (!empty($this->folders)) {
                foreach ($this->folders as $unique_key => $folder) {
                    if (strpos($unique_key, "DAV:$type:") === 0) {
                        $folder_data = $this->folder_data($folder, $type);
                        $list[$folder_data['serverId']] = $folder_data;
                    }
                }
            }

            if (empty($list)) {
                foreach ($this->subscriptions->list_subscriptions($deviceid, $type) as $folder) {
                    /** @var kolab_storage_dav_folder $folder */
                    $folder = $folder[2];
                    $folder_data = $this->folder_data($folder, $type);
                    $list[$folder_data['serverId']] = $folder_data;

                    // Store all folder objects in internal cache, otherwise
                    // Any access to the folder (or list) will invoke excessive DAV requests
                    $unique_key = $folder_data['serverId'] . ":$deviceid:$type";
                    $this->folders[$unique_key] = $folder;
                }
            }
        }

        return $list;
    }

    /**
     * Converts list of mail folders to a "flat" list
     */
    private function folders_list_flat_mail($folders)
    {
        $delim = $this->storage->get_hierarchy_delimiter();

        // for mail folders we modify only folders with non-existing parents
        foreach ($folders as $idx => $folder) {
            if ($folder['parentId'] && !isset($folders[$folder['parentId']])) {
                $items  = explode($delim, $folder['imap_name']);
                $parent = 0;
                $parent_id = null;

                // find existing parent
                while (count($items) > 0) {
                    array_pop($items);
                    $parent_id = $this->folder_id(implode($delim, $items), 'mail');
                    if (isset($folders[$parent_id])) {
                        $parent = $parent_id;
                        break;
                    }
                }

                // Modify the folder name to include "path"
                if (!$parent) {
                    $display_name = kolab_storage::object_prettyname($folder['imap_name']);
                    $display_name = html_entity_decode($display_name, ENT_COMPAT, RCUBE_CHARSET);
                } else {
                    $parent_name  = isset($parent_id) ? $folders[$parent_id]['imap_name'] : '';
                    $display_name = substr($folder['imap_name'], strlen($parent_name) + 1);
                    $display_name = rcube_charset::convert($display_name, 'UTF7-IMAP');
                    $display_name = str_replace($delim, ' » ', $display_name);
                }

                $folders[$idx]['parentId']    = $parent;
                $folders[$idx]['displayName'] = $display_name;
            }
        }

        return $folders;
    }

    /**
     * Creates folder and subscribes to the device
     *
     * @param string  $name      Folder name (UTF8)
     * @param int     $type      Folder (ActiveSync) type
     * @param string  $deviceid  Device identifier
     * @param ?string $parentid  Parent folder identifier
     *
     * @return string|false New folder identifier on success, False on failure
     */
    public function folder_create($name, $type, $deviceid, $parentid = null)
    {
        // Mail folder
        if ($type <= 6 || $type == 12) {
            $parent = null;
            $name = rcube_charset::convert($name, kolab_sync::CHARSET, 'UTF7-IMAP');

            if ($parentid) {
                $parent = $this->folder_id2name($parentid, $deviceid, self::MODEL_EMAIL);

                if ($parent === null) {
                    throw new Syncroton_Exception_Status_FolderCreate(Syncroton_Exception_Status_FolderCreate::PARENT_NOT_FOUND);
                }
            }

            if ($parent !== null) {
                $delim = $this->storage->get_hierarchy_delimiter();
                $name  = $parent . $delim . $name;
            }

            if ($this->storage->folder_exists($name)) {
                throw new Syncroton_Exception_Status_FolderCreate(Syncroton_Exception_Status_FolderCreate::FOLDER_EXISTS);
            }

            // TODO: Support setting folder types?

            $created = $this->storage->create_folder($name, true);

            if ($created) {
                // Set ActiveSync subscription flag
                $this->subscriptions->folder_subscribe($deviceid, $name, 1, self::MODEL_EMAIL);

                return $this->folder_id($name, 'mail');
            }

            // Special case when client tries to create a subfolder of INBOX
            // which is not possible on Cyrus-IMAP (T2223)
            if ($parent == 'INBOX' && stripos($this->last_error(), 'invalid') !== false) {
                throw new Syncroton_Exception('', Syncroton_Exception_Status_FolderCreate::SPECIAL_FOLDER);
            }

            return false;
        } elseif ($type == 8 || $type == 13 || $type == 7 || $type == 15 || $type == 9 || $type == 14) {
            // DAV folder
            $type = preg_replace('|\..*|', '', self::type_activesync2kolab($type));

            // TODO: Folder hierarchy support

            // Check if folder exists
            foreach ($this->davStorage->get_folders($type) as $folder) {
                if ($folder->get_name() == $name) {
                    throw new Syncroton_Exception_Status_FolderCreate(Syncroton_Exception_Status_FolderCreate::FOLDER_EXISTS);
                }
            }

            $props = ['name' => $name, 'type' => $type];

            if ($id = $this->davStorage->folder_update($props)) {
                // Set ActiveSync subscription flag
                $this->subscriptions->folder_subscribe($deviceid, $this->davStorage->new_location, 1, $type);
                $this->folders = [];

                return "DAV:{$type}:{$id}";
            }

            return false;
        }

        throw new \Exception("Not implemented");
    }

    /**
     * Renames a folder
     *
     * @param string  $folderid Folder identifier
     * @param string  $deviceid Device identifier
     * @param string  $type     Activesync model name (folder type)
     * @param string  $new_name New folder name (UTF8)
     * @param ?string $parentid Folder parent identifier
     *
     * @return bool True on success, False on failure
     */
    public function folder_rename($folderid, $deviceid, $type, $new_name, $parentid)
    {
        // DAV folder
        if (strpos($folderid, 'DAV:') === 0) {
            [, $type, $id] = explode(':', $folderid);
            $props = [
                'id' => $id,
                'name' => $new_name,
                'type' => $type,
            ];

            // TODO: Folder hierarchy support

            return $this->davStorage->folder_update($props) !== false;
        }

        // Mail folder
        $old_name = $this->folder_id2name($folderid, $deviceid, $type);

        if ($parentid) {
            $parent = $this->folder_id2name($parentid, $deviceid, $type);
        }

        $name = rcube_charset::convert($new_name, kolab_sync::CHARSET, 'UTF7-IMAP');

        if (isset($parent)) {
            $delim = $this->storage->get_hierarchy_delimiter();
            $name  = $parent . $delim . $name;
        }

        if ($name === $old_name) {
            return true;
        }

        $result = $this->storage->rename_folder($old_name, $name);

        if ($result) {
            // Set ActiveSync subscription flag
            $this->subscriptions->folder_subscribe($deviceid, $name, 1, self::MODEL_EMAIL);
        }

        return $result;
    }

    /**
     * Deletes folder
     *
     * @param string $folderid Folder identifier
     * @param string $deviceid Device identifier
     * @param string $type     Activesync model name (folder type)
     *
     * @return bool True on success, False otherwise
     */
    public function folder_delete($folderid, $deviceid, $type)
    {
        // DAV folder
        if (strpos($folderid, 'DAV:') === 0) {
            [, $type, $id] = explode(':', $folderid);

            return $this->davStorage->folder_delete($id, $type) !== false;
        }

        // Mail folder
        $name = $this->folder_id2name($folderid, $deviceid, $type);

        return $this->storage->delete_folder($name);
    }

    /**
     * Deletes contents of a folder
     *
     * @param string $folderid  Folder identifier
     * @param string $deviceid  Device identifier
     * @param string $type      Activesync model name (folder type)
     * @param bool   $recursive Apply to the folder and its subfolders
     *
     * @return bool True on success, False otherwise
     */
    public function folder_empty($folderid, $deviceid, $type, $recursive = false)
    {
        // DAV folder
        if (strpos($folderid, 'DAV:') === 0) {
            [, $type, $id] = explode(':', $folderid);

            if ($folder = $this->davStorage->get_folder($id, $type)) {
                return $folder->delete_all();
            }

            // TODO: $recursive=true

            return false;
        }

        // Mail folder
        return parent::folder_empty($folderid, $deviceid, $type, $recursive);
    }

    /**
     * Returns folder data in Syncroton format
     */
    protected function folder_data($folder, $type)
    {
        // Mail folders
        if (strpos($type, 'mail') === 0) {
            return parent::folder_data($folder, $type);
        }

        // DAV folders
        return [
            'serverId'    => "DAV:{$type}:{$folder->id}",
            'parentId'    => 0, // TODO: Folder hierarchy
            'displayName' => $folder->get_name(),
            'type'        => $this->type_kolab2activesync($folder->default ? "$type.default" : $type),
        ];
    }

    /**
     * Builds folder ID based on folder name
     *
     * @param string $name Folder name (UTF7-IMAP)
     * @param string $type Kolab folder type
     *
     * @return string|null Folder identifier (up to 64 characters)
     */
    protected function folder_id($name, $type = null)
    {
        if (!$type) {
            $type = 'mail';
        }

        // ActiveSync expects folder identifiers to be max.64 characters
        // So we can't use just folder name
        $name = (string) $name;

        if ($name === '') {
            return null;
        }

        if (strpos($type, 'mail') !== 0) {
            throw new Exception("Unsupported folder_id() call on a DAV folder");
        }

        if (isset($this->folder_uids[$name])) {
            return $this->folder_uids[$name];
        }

        if (strcasecmp($name, 'INBOX') === 0) {
            // INBOX is always inbox, prevent from issues related with a change of
            // folder type annotation (it can be initially unset).
            $type = 'mail.inbox';
        }

        // Add type to folder UID hash, so type change can be detected by Syncroton
        $uid = $name . '!!' . $type;
        $uid = md5($uid);

        return $this->folder_uids[$name] = $uid;
    }

    /**
     * Returns IMAP folder name
     *
     * @param string $id       Folder identifier
     * @param string $deviceid Device dentifier
     * @param string $type     Folder type
     *
     * @return null|string Folder name (UTF7-IMAP)
     */
    public function folder_id2name($id, $deviceid, $type)
    {
        if (strpos($id, 'DAV:') === 0) {
            throw new Exception("Unsupported folder_id2name() call on a DAV folder");
        }

        return parent::folder_id2name($id, $deviceid, $type);
    }

    /**
     * Returns list of categories assigned to an object
     *
     * @param rcube_message_header|string $object     UID or mail message headers
     * @param array                       $categories Addition tag names to merge with
     *
     * @return array List of categories
     */
    protected function getCategories($object, $categories = [])
    {
        $tags = $this->tagStorage->get_tags_for($object);

        // merge result with old categories
        if (!empty($categories)) {
            $tags = array_unique(array_merge($tags, (array) $categories));
        }

        return $tags;
    }

    /**
     * Detect changes of tag objects data and assigned messages
     */
    protected function getChangesByRelations($folderid, $device_key, $type, $filter)
    {
        // This is not needed with Kolab4 that uses METADATA/ANNOTATE and immutable tags
        return [];
    }

    /**
     * Gets kolab_storage_folder object from Activesync folder ID.
     *
     * @param string $folderid Folder identifier
     * @param string $deviceid Device identifier
     * @param string $type     Activesync model name (folder type)
     *
     * @return ?kolab_storage_dav_folder
     */
    public function getFolder($folderid, $deviceid, $type)
    {
        if (strpos($folderid, 'DAV:') !== 0) {
            throw new Exception("Unsupported getFolder() call on a mail folder");
        }

        $unique_key = "$folderid:$deviceid:$type";

        if (array_key_exists($unique_key, $this->folders)) {
            return $this->folders[$unique_key];
        }

        [, $type, $id] = explode(':', $folderid);

        return $this->folders[$unique_key] = $this->davStorage->get_folder($id, $type);
    }

    /**
     * Gets Activesync preferences for a folder.
     *
     * @param string $folderid Folder identifier
     * @param string $deviceid Device identifier
     * @param string $type     Activesync model name (folder type)
     *
     * @return array Folder preferences
     */
    public function getFolderConfig($folderid, $deviceid, $type)
    {
        $alarms = 0;

        if ($folder = $this->getFolder($folderid, $deviceid, $type)) {
            $subs = $this->subscriptions->folder_subscriptions($folder->href, $type);
            $alarms = $subs[$deviceid] ?? 0;
        }

        return [
            'ALARMS' => $alarms == 2,
        ];
    }

    /**
     * Return last storage error
     */
    public static function last_error()
    {
        // TODO
        return null;
    }

    /**
     * Subscribes to a default set of folder on a new device registration
     *
     * @param string $deviceid Device ID
     */
    public function device_init($deviceid)
    {
        $config  = rcube::get_instance()->config;
        $mode    = (int) $config->get('activesync_init_subscriptions');

        $subscribed_folders = null;

        // Special folders only
        if (!$mode) {
            $all_folders = $this->storage->get_special_folders(true);
            // We do not subscribe to the Spam folder by default, same as the old Kolab driver does
            unset($all_folders['junk']);
            $all_folders = array_unique(array_merge(['INBOX'], array_values($all_folders)));
        }
        // other modes
        elseif (($mode & self::INIT_ALL_PERSONAL) || ($mode & self::INIT_ALL_OTHER) || ($mode & self::INIT_ALL_SHARED)) {
            $all_folders = $this->storage->list_folders();
            if (($mode & self::INIT_SUB_PERSONAL) || ($mode & self::INIT_SUB_OTHER) || ($mode & self::INIT_SUB_SHARED)) {
                $subscribed_folders = $this->storage->list_folders_subscribed();
            }
        } else {
            $all_folders = $this->storage->list_folders_subscribed();
        }

        $subscribe = [
            'mail' => ['INBOX' => 1],
            'contact' => [],
            'event' => [],
            'task' => [],
        ];

        foreach ($all_folders as $folder) {
            $ns = strtoupper($this->storage->folder_namespace($folder));

            // subscribe the folder according to configured mode
            // and folder namespace/subscription status
            if (!$mode
                || ($mode & constant("self::INIT_ALL_{$ns}"))
                || (($mode & constant("self::INIT_SUB_{$ns}")) && ($subscribed_folders === null || in_array($folder, $subscribed_folders)))
            ) {
                $subscribe['mail'][$folder] = 1;
            }
        }

        foreach ($subscribe as $type => $list) {
            if ($type != 'mail') {
                foreach ($this->subscriptions->list_folders($type) as $folder) {
                    // TODO: Subscribe personal DAV folders, for now we assume all are subscribed
                    $list[$folder[0]] = ($type == 'event' || $type == 'task') ? 2 : 1;
                }
            }

            $this->subscriptions->set_subscriptions($deviceid, $type, $list);
        }
    }

    public function getExtraData($folderid, $deviceid, $type)
    {
        if (strpos($folderid, 'DAV:') === 0) {
            return null;
        }

        return parent::getExtraData($folderid, $deviceid, $type);
    }

    /**
     * Set categories to an object
     *
     * @param rcube_message_header|string $object     UID or mail message headers
     * @param array                       $categories List of Category names
     */
    protected function setCategories($object, $categories)
    {
        $this->tagStorage->set_tags_for($object, $categories);
    }
}
