#!/usr/bin/env php
<?php

/*
 +--------------------------------------------------------------------------+
 | Kolab Sync (ActiveSync for Kolab)                                        |
 |                                                                          |
 | Copyright (C) 2024, Apheleia IT AG <contact@apheleia-it.ch>         |
 |                                                                          |
 | This program is free software: you can redistribute it and/or modify     |
 | it under the terms of the GNU Affero General Public License as published |
 | by the Free Software Foundation, either version 3 of the License, or     |
 | (at your option) any later version.                                      |
 |                                                                          |
 | This program is distributed in the hope that it will be useful,          |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of           |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the             |
 | GNU Affero General Public License for more details.                      |
 |                                                                          |
 | You should have received a copy of the GNU Affero General Public License |
 | along with this program. If not, see <http://www.gnu.org/licenses/>      |
 +--------------------------------------------------------------------------+
 | Author: Christian Mollekopf <mollekopf@apheleia-it.ch>                      |
 +--------------------------------------------------------------------------+
*/


require_once 'shared.php';

function filterTypeToIMAPSearch($filter_type = 0)
{
    switch ($filter_type) {
        case 1:
            $mod = '-1 day';
            break;
        case 2:
            $mod = '-3 days';
            break;
        case 3:
            $mod = '-1 week';
            break;
        case 4:
            $mod = '-2 weeks';
            break;
        case 5:
            $mod = '-1 month';
            break;
    }

    if (!empty($mod)) {
        $dt = new DateTime('now', new DateTimeZone('UTC'));
        $dt->modify($mod);
        // RFC3501: IMAP SEARCH
        return 'SINCE ' . $dt->format('d-M-Y');
    }

    return "";
}


$opts = rcube_utils::get_opt([
    'e' => 'email',
    'p' => 'adminpassword',
    'd' => 'debug',
    'k' => 'dump',
]);

if (empty($opts['email'])) {
    rcube::raise_error("Email address not specified (--email).", false, true);
}
$email = $opts['email'];

$cli = new SyncrotonCli();
$userid = $cli->selectUser($opts['email'] ?? null);
$db = $cli->db;
$debug = !empty($opts['debug']);
$imap = $cli->connectToImap($opts['email'] ?? null, $opts['password'] ?? null, $opts['adminpassword'] ?? null, $debug);

$devicesSelect = $db->query(
    "SELECT `id`, `deviceid`, `devicetype` FROM `syncroton_device`"
    . " WHERE `owner_id` = ?",
    $userid
);

$result = [];
while ($data = $db->fetch_assoc($devicesSelect)) {
    $deviceid = $data["deviceid"];
    $device_id = $data["id"];

    $result[$device_id]['deviceid'] = $deviceid;
    $result[$device_id]['device_id'] = $device_id;
    $result[$device_id]['devicetype'] = $data["devicetype"];

    $select = $db->limitquery(
        "SELECT `counter`, `lastsync` FROM `syncroton_synckey`"
        . " WHERE `device_id` = ? AND `type` = 'FolderSync'"
        . " ORDER BY `counter` DESC",
        0,
        1,
        $device_id
    );

    if ($data = $db->fetch_assoc($select)) {
        $result[$device_id]['FolderSync'] = [
            "counter" => $data['counter'],
            "lastsync" => $data['lastsync'],
        ];
    } else {
        echo("Synckey not found.\n");
    }

    $folderSelect = $db->query(
        "SELECT * FROM `syncroton_folder`"
        . " WHERE `device_id` = ?",
        $device_id
    );

    while ($folder = $db->fetch_assoc($folderSelect)) {
        $select = $db->limitquery(
            "SELECT `counter`, `lastsync`, `extra_data` FROM `syncroton_synckey`"
            . " WHERE `device_id` = ? AND `type` = ?"
            . " ORDER BY `counter` DESC",
            0,
            1,
            $device_id,
            $folder['id']
        );

        if ($data = $db->fetch_assoc($select)) {
            $result[$device_id]['folders'][$folder['id']] = [
                "counter" => $data['counter'],
                "lastsync" => $data['lastsync'],
                "modseq" => $data['extra_data'] ? json_decode($data['extra_data'])->modseq : null,
            ];
        }

        $result[$device_id]['folders'][$folder['id']]['name'] = $folder['displayname'];
        $result[$device_id]['folders'][$folder['id']]['folderid'] = $folder['folderid'];
        $result[$device_id]['folders'][$folder['id']]['class'] = $folder['class'];
        $result[$device_id]['folders'][$folder['id']]['lastfiltertype'] = $folder['lastfiltertype'] ?? null;

        if ($imap->select($folder['displayname'])) {
            $result[$device_id]['folders'][$folder['id']]['imapModseq'] = $imap->data['HIGHESTMODSEQ'] ?? null;
            $index = $imap->search(
                $folder['displayname'],
                'ALL UNDELETED ' . filterTypeToIMAPSearch($folder['lastfiltertype']),
                false,
                ['COUNT']
            );
            if (!$index->is_error()) {
                $result[$device_id]['folders'][$folder['id']]['imapMessagecount'] = $index->count();
            }
        } else {
            $result[$device_id]['folders'][$folder['id']]['imapDeleted'] = true;
        }

        $select = $db->query(
            "SELECT count(*) FROM `syncroton_content`"
            . " WHERE `device_id` = ? AND `folder_id` = ?",
            $device_id,
            $folder['id']
        );

        if ($data = $db->fetch_assoc($select)) {
            $result[$device_id]['folders'][$folder['id']]['contentCount'] = array_values($data)[0];
        }
    }
}

if ($debug) {
    var_export($result);
}

function println($output)
{
    print("{$output}\n");
}

function filterType($value)
{
    if (!$value) {
        return "No filter";
    }
    switch ($value) {
        case 0: return "No filter";
        case 1: return "1 day";
        case 2: return "3 days";
        case 3: return "1 week";
        case 4: return "2 weeks";
        case 5: return "1 month";
        case 6: return "3 months (WARNING: not implemented)";
        case 7: return "6 months (WARNING: not implemented)";
        case 8: return "Filter by incomplete tasks";
    }
    return "Unknown value: $value";
}


function getContentIds($db, $device_id, $folder_id)
{
    $contentSelect = $db->query(
        "SELECT contentid FROM `syncroton_content`"
        . " WHERE `device_id` = ? AND `folder_id` = ? AND `is_deleted` = 0",
        $device_id,
        $folder_id
    );

    $contentUids = [];
    while ($content = $db->fetch_assoc($contentSelect)) {
        $contentUids[] = $content['contentid'];
    }
    return $contentUids;
}

function getContentUids($db, $device_id, $folder_id)
{
    $contentSelect = $db->query(
        "SELECT contentid FROM `syncroton_content`"
        . " WHERE `device_id` = ? AND `folder_id` = ? AND `is_deleted` = 0",
        $device_id,
        $folder_id
    );

    $contentUids = [];
    while ($content = $db->fetch_assoc($contentSelect)) {
        $contentUids[] = explode('::', $content['contentid'])[1];
    }
    return $contentUids;
}

function getImapUids($imap, $folder, $lastfiltertype)
{
    $imap->select($folder);
    $index = $imap->search($folder, 'ALL UNDELETED ' . filterTypeToIMAPSearch($lastfiltertype), true);
    if (!$index->is_error()) {
        return $index->get();
    }
    return [];
}

println("");
foreach ($result as $deviceId => $values) {
    $device_id = $values['deviceid'];
    println("Device: $device_id ($deviceId)");
    println("  Devicetype: " . $values['devicetype']);
    if (empty($values['FolderSync'])) {
        println("  Folders never synced.");
        println("");
        continue;
    } else {
        println("  Last folder sync: " . $values['FolderSync']['lastsync']);
        println("  Folder sync count: " . $values['FolderSync']['counter']);
    }
    println("  Folders:");
    foreach ($values['folders'] ?? [] as $folderId => $folder) {
        $folder_id = $folder['folderid'];
        println("    " . $folder['name'] . " $folder_id ($folderId)");
        $messageCount = $folder['contentCount'];
        $totalCount = $folder['imapMessagecount'] ?? "unknown";
        $modseq = $folder['modseq'] ?? "none";

        if ($folder['imapDeleted'] ?? null) {
            println("    Status: Folder removed in imap");
            println("");
            continue;
        }

        $imapModseq = $folder['imapModseq'];
        // We're not using modseq for groupware folders
        if ($messageCount == $totalCount && ($modseq == "none" || $modseq == $imapModseq)) {
            println("    Status: Fully Synced ($messageCount messages)");
        } else {
            println("    Status: Incomplete ($messageCount/$totalCount messages)");
            println("    Modseq: " . $modseq . "/" . $imapModseq);
        }
        println("    Last sync: " . ($folder['lastsync'] ?? "None"));
        println("    Number of syncs: " . ($folder['counter'] ?? "None"));
        println("    Filter type: " . filterType($folder['lastfiltertype'] ?? null));


        if (!empty($opts['dump']) && $opts['dump'] == $folder['name']) {
            $contentUids = getContentUids($db, $deviceId, $folderId);
            $imapUids = getImapUids($imap, $folder['name'], $folder['lastfiltertype'] ?? null);

            $entries = array_diff($imapUids, $contentUids);
            if (!empty($entries)) {
                println("       The following messages are on the server, but not the device:");
                foreach ($entries as $uid) {
                    println("           $uid");
                    //TODO get details from imap?
                }
            }

            $entries = array_diff($contentUids, $imapUids);
            if (!empty($entries)) {
                println("       The following messages are on the device, but not the server:");
                foreach ($entries as $uid) {
                    println("           $uid");
                    //TODO get details from the content part?
                    //TODO display creation_synckey?
                }
            }

            $contentUids = getContentUids($db, $deviceId, $folderId);
            foreach (array_slice($contentUids, -10, 10) as $uid) {
                println($uid);
            }

            $contentIds = getContentIds($db, $deviceId, $folderId);
            foreach (array_slice($contentIds, -10, 10) as $uid) {
                println($uid);
            }
        }

        if (($folder['class'] == "Email") && ($folder['counter'] ?? false) && $messageCount != $totalCount && ($modseq == "none" || $modseq == $imapModseq)) {
            if (($folder['lastfiltertype'] ?? false) && $messageCount > $totalCount) {
                // This doesn't have to indicate an issue, since the timewindow of the filter wanders, so some messages that have been synchronized may no longer match the window.
            } else {
                println("    Issue Detected: The sync state seems to be inconsistent. The device should be fully synced, but the sync counts differ.");
                println("    There are $messageCount ContentParts (should match number of messages on the device), but $totalCount messages in IMAP matching the filter.");

                $contentUids = getContentUids($db, $deviceId, $folderId);
                $imapUids = getImapUids($imap, $folder['name'], $folder['lastfiltertype'] ?? null);

                $entries = array_diff($imapUids, $contentUids);
                if (!empty($entries)) {
                    println("       The following messages are on the server, but not the device:");
                    foreach ($entries as $uid) {
                        println("           $uid");
                        //TODO get details from imap?
                    }
                }

                $entries = array_diff($contentUids, $imapUids);
                if (!empty($entries)) {
                    println("       The following messages are on the device, but not the server:");
                    foreach ($entries as $uid) {
                        println("           $uid");
                        //TODO get details from the content part?
                        //TODO display creation_synckey?
                    }
                }
                println("");
            }
        }

        println("");
    }
}
