<?php

/**
 * Syncroton
 *
 * @package     Wbxml
 * @subpackage  Wbxml
 * @license     http://www.tine20.org/licenses/lgpl.html LGPL Version 3
 * @copyright   Copyright (c) 2008-2009 Metaways Infosystems GmbH (http://www.metaways.de)
 * @author      Lars Kneschke <l.kneschke@metaways.de>
 * @version     $Id:Encoder.php 4968 2008-10-17 09:09:33Z l.kneschke@metaways.de $
 */

/**
 * class to convert XML to WBXML
 *
 * @package     Wbxml
 * @subpackage  Wbxml
 */

class Syncroton_Wbxml_Encoder extends Syncroton_Wbxml_Abstract
{
    /**
     * count level of tags
     *
     * @var int
     */
    protected $_level = 0;

    /**
     * the constructor
     *
     * @param resource $_stream
     * @param string   $_charSet
     * @param int      $_version
     */
    public function __construct($_stream, $_charSet = 'UTF-8', $_version = 2)
    {
        $this->_stream = $_stream;
        $this->_charSet = $_charSet;
        $this->_version = $_version;
    }

    /**
     * initialize internal variables and write wbxml header to stream
     *
     * @param DOMDocument $_dom
     * @todo check if dpi > 0, instead checking the urn
     */
    protected function _initialize($_dom)
    {
        $this->_dtd = Syncroton_Wbxml_Dtd_Factory::factory($_dom->doctype->name);
        $this->_codePage = $this->_dtd->getCurrentCodePage();

        // the WBXML version
        $this->_writeByte($this->_version);

        if ($this->_codePage->getDPI() === null) {
            // the document public identifier
            $this->_writeMultibyteUInt(1);
        } else {
            // the document public identifier
            // defined in string table
            $this->_writeMultibyteUInt(0);
            // the offset of the DPI in the string table
            $this->_writeByte(0);
        }

        // write the charSet
        $this->_writeCharSet($this->_charSet);

        if ($this->_codePage->getDPI() === null) {
            // the length of the string table
            $this->_writeMultibyteUInt(0);
        } else {
            // the length of the string table
            $this->_writeMultibyteUInt(strlen($this->_codePage->getDPI()));
            // the dpi
            $this->_writeString($this->_codePage->getDPI());
        }
    }

    /**
     * write charset to stream
     *
     * @param string $_charSet
     * @todo add charset lookup table. currently only utf-8 is supported
     */
    protected function _writeCharSet($_charSet)
    {
        switch (strtoupper($_charSet)) {
            case 'UTF-8':
                $this->_writeMultibyteUInt(106);
                break;

            default:
                throw new Syncroton_Wbxml_Exception('unsuported charSet ' . strtoupper($_charSet));
        }
    }

    /**
     * start encoding of xml to wbxml
     *
     * @param DOMDocument $_dom the DOM document
     */
    public function encode(DOMDocument $_dom)
    {
        $_dom->formatOutput = false;

        $this->_initialize($_dom);
        $this->_traverseDom($_dom);
    }

    private function getAttributes($node)
    {
        $attributes = [];
        if ($node->attributes) {
            for ($i = 0; $i < $node->attributes->length; ++$i) {
                $attributes[$node->attributes->item($i)->name] = $node->attributes->item($i)->value;
            }
        }
        return $attributes;
    }

    private function writeNode($node, $withContent = false, $data = null)
    {
        if ($this->_codePage->getNameSpace() != $node->namespaceURI) {
            $this->_switchCodePage($node->namespaceURI);
        }
        $this->_writeTag($node->localName, $this->getAttributes($node), $withContent, $data);
    }

    protected function _traverseDom($_dom)
    {
        if ($_dom->childNodes->length == 0) {
            return false;
        }
        // print(str_pad("", $this->_level, " ") . "traversing {$_dom->nodeName}" . "\n");
        $this->_level++;
        $prevNode = $_dom;
        $foundElementNode = false;
        foreach ($_dom->childNodes as $node) {
            if ($node->nodeType == XML_ELEMENT_NODE) {
                $foundElementNode = true;
                if ($prevNode && $this->_level > 1) {
                    // print(str_pad("", $this->_level, " ") . "{$node->nodeName} creating parent {$prevNode->nodeName}" . "\n");
                    $this->writeNode($prevNode, true);
                    $prevNode = null;
                }
                if (!$this->_traverseDom($node)) {
                    // print(str_pad("", $this->_level, " ") . "{$node->nodeName} content {$node->nodeValue}" . "\n");
                    $data = $node->nodeValue;
                    if (strlen($data) == 0) {
                        $this->writeNode($node);
                    } else {
                        $this->writeNode($node, true, $data);
                        $this->_writeByte(Syncroton_Wbxml_Abstract::END);
                        // print("Closing tag after writing tag\n");
                    }
                } else {
                    $this->_writeByte(Syncroton_Wbxml_Abstract::END);
                    // print("Closing tag\n");
                }
            }
        }
        $this->_level--;

        return $foundElementNode;
    }

    /**
     * strip uri: from nameSpace
     *
     * @param string $_nameSpace
     *
     * @return string
     */
    protected function _stripNameSpace($_nameSpace)
    {
        return substr($_nameSpace, 4);
    }

    /**
     * writes tag with data to stream
     *
     * @param string $_tag
     * @param array $_attributes
     * @param bool $_hasContent
     * @param string $_data
     */
    protected function _writeTag($_tag, $_attributes = null, $_hasContent = false, $_data = null)
    {
        if ($_hasContent == false && $_data !== null) {
            throw new Syncroton_Wbxml_Exception('$_hasContent can not be false, when $_data !== NULL');
        }

        // handle the tag
        $identity = $this->_codePage->getIdentity($_tag);

        if (is_array($_attributes) && isset($_attributes['uri:Syncroton;encoding'])) {
            $encoding = 'opaque';
            unset($_attributes['uri:Syncroton;encoding']);
        } else {
            $encoding = 'termstring';
        }

        if (!empty($_attributes)) {
            $identity |= 0x80;
        }

        if ($_hasContent == true) {
            $identity |= 0x40;
        }

        $this->_writeByte($identity);

        // handle the data
        if ($_data !== null) {
            if ($encoding == 'opaque') {
                $this->_writeOpaqueString(base64_decode($_data));
            } else {
                $this->_writeTerminatedString($_data);
            }
        }
    }

    /**
     * switch code page
     *
     * @param string $_nameSpace
     */
    protected function _switchCodePage($_nameSpace)
    {
        $codePageName = $this->_stripNameSpace($_nameSpace);
        if (!defined('Syncroton_Wbxml_Dtd_ActiveSync::CODEPAGE_' . strtoupper($codePageName))) {
            throw new Syncroton_Wbxml_Exception('codepage ' . $codePageName . ' not found');
        }
        // switch to another codepage
        // no need to write the wbxml header again
        $codePageId = constant('Syncroton_Wbxml_Dtd_ActiveSync::CODEPAGE_' . strtoupper($codePageName));
        $this->_codePage = $this->_dtd->switchCodePage($codePageId);

        $this->_writeByte(Syncroton_Wbxml_Abstract::SWITCH_PAGE);
        $this->_writeByte($codePageId);
    }
}
