<?php

namespace Tests\Sync\Sync;

class ContactsTest extends \Tests\SyncTestCase
{
    /**
     * Test Sync command
     */
    public function testSync()
    {
        $davFolder = 'Contacts';
        $this->emptyTestFolder($davFolder, 'contact');
        $this->deleteTestFolder('Test Contacts Folder', 'contact'); // from other test files
        $this->registerDevice();

        // Test empty contacts folder
        $folderId = 'Contacts::Syncroton';
        $syncKey = 0;
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase">
                <Collections>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>{$syncKey}</SyncKey>
                        <CollectionId>{$folderId}</CollectionId>
                        <DeletesAsMoves/>
                        <GetChanges/>
                        <Options>
                            <AirSyncBase:BodyPreference>
                                <AirSyncBase:Type>1</AirSyncBase:Type>
                                <AirSyncBase:TruncationSize>5120</AirSyncBase:TruncationSize>
                            </AirSyncBase:BodyPreference>
                            <Conflict>1</Conflict>
                        </Options>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = "//ns:Sync/ns:Collections/ns:Collection";
        $this->assertSame('1', $xpath->query("{$root}/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("{$root}/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame($folderId, $xpath->query("{$root}/ns:CollectionId")->item(0)->nodeValue);
        $this->assertSame(0, $xpath->query("{$root}/ns:Commands/ns:Add")->count());

        if ($this->isStorageDriver('kolab')) {
            $this->markTestSkipped("Appending test objects does not work with 'kolab' storage yet.");
        }

        // Append two contact objects and sync them
        // TODO: Test a folder with contact groups inside
        $this->appendObject($davFolder, 'contact.vcard1', 'contact');
        $this->appendObject($davFolder, 'contact.vcard2', 'contact');

        $request = str_replace("<SyncKey>0</SyncKey>", "<SyncKey>{$syncKey}</SyncKey>", $request);

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = "//ns:Sync/ns:Collections/ns:Collection";
        $this->assertSame('1', $xpath->query("{$root}/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("{$root}/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame($folderId, $xpath->query("{$root}/ns:CollectionId")->item(0)->nodeValue);
        $this->assertSame(2, $xpath->query("{$root}/ns:Commands/ns:Add")->count());

        $root .= "/ns:Commands/ns:Add";
        $this->assertStringMatchesFormat("CRC%s", $xpath->query("{$root}/ns:ServerId")->item(0)->nodeValue);
        $r = "{$root}[1]/ns:ApplicationData";
        $this->assertSame('Jack', $xpath->query("{$r}/Contacts:FirstName")->item(0)->nodeValue);
        $this->assertSame('Strong', $xpath->query("{$r}/Contacts:LastName")->item(0)->nodeValue);
        $this->assertSame('jack@kolab.org', $xpath->query("{$r}/Contacts:Email1Address")->item(0)->nodeValue);
        $this->assertNull($xpath->query("{$r}/Contacts:Email2Address")->item(0));
        $this->assertNull($xpath->query("{$r}/Contacts:Email3Address")->item(0));
        $r = "{$root}[2]/ns:ApplicationData";
        $this->assertSame('Jane', $xpath->query("{$r}/Contacts:FirstName")->item(0)->nodeValue);
        $this->assertSame('J.', $xpath->query("{$r}/Contacts:MiddleName")->item(0)->nodeValue);
        $this->assertSame('Doe', $xpath->query("{$r}/Contacts:LastName")->item(0)->nodeValue);
        $this->assertNull($xpath->query("{$r}/Contacts:Email1Address")->item(0));
        $this->assertNull($xpath->query("{$r}/Contacts:Email2Address")->item(0));
        $this->assertNull($xpath->query("{$r}/Contacts:Email3Address")->item(0));

        return $syncKey;
    }

    /**
     * Test adding objects from client
     *
     * @depends testSync
     */
    public function testAddFromClient($syncKey)
    {
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase" xmlns:Contacts="uri:Contacts">
                <Collections>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>{$syncKey}</SyncKey>
                        <CollectionId>Contacts::Syncroton</CollectionId>
                        <DeletesAsMoves/>
                        <GetChanges/>
                        <Options>
                            <AirSyncBase:BodyPreference>
                                <AirSyncBase:Type>1</AirSyncBase:Type>
                                <AirSyncBase:TruncationSize>5120</AirSyncBase:TruncationSize>
                            </AirSyncBase:BodyPreference>
                            <Conflict>1</Conflict>
                        </Options>
                        <Commands>
                            <Add>
                                <ClientId>42</ClientId>
                                <ApplicationData>
                                    <Contacts:FirstName>Lars</Contacts:FirstName>
                                    <Contacts:LastName>Ulrich</Contacts:LastName>
                                    <Contacts:Email1Address>lars@kolab.org</Contacts:Email1Address>
                                    <Contacts:Email2Address>lars.tw@kolab.org</Contacts:Email2Address>
                                    <Contacts:Email3Address>lars.th@kolab.org</Contacts:Email3Address>
                                </ApplicationData>
                            </Add>
                        </Commands>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(0);
        $this->assertSame('1', $xpath->query("ns:Status", $root)->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("ns:SyncKey", $root)->item(0)->nodeValue);
        $root = $xpath->query("ns:Responses/ns:Add", $root)->item(0);
        $this->assertSame('1', $xpath->query("ns:Status", $root)->item(0)->nodeValue);
        $this->assertSame('42', $xpath->query("ns:ClientId", $root)->item(0)->nodeValue);
        $serverId = $xpath->query("ns:ServerId", $root)->item(0)->nodeValue;
        $this->assertStringMatchesFormat("CRC%s", $serverId);

        // Assert the content on the server
        $contacts = $this->getDavObjects('Contacts', 'contact');
        $this->assertCount(3, $contacts);
        usort($contacts, function ($c1, $c2) { return $c1['surname'] <=> $c2['surname']; });
        $this->assertSame('Lars', $contacts[2]['firstname']);
        $this->assertSame('Ulrich', $contacts[2]['surname']);
        $this->assertSame(['lars@kolab.org', 'lars.tw@kolab.org', 'lars.th@kolab.org'], $contacts[2]['email:other']);

        return [$syncKey, $serverId];
    }

    /**
     * Test updating objects from client
     *
     * @depends testAddFromClient
     */
    public function testChangeFromClient($params)
    {
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase" xmlns:Contacts="uri:Contacts">
                <Collections>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>{$params[0]}</SyncKey>
                        <CollectionId>Contacts::Syncroton</CollectionId>
                        <DeletesAsMoves/>
                        <GetChanges/>
                        <Options>
                            <AirSyncBase:BodyPreference>
                                <AirSyncBase:Type>1</AirSyncBase:Type>
                                <AirSyncBase:TruncationSize>5120</AirSyncBase:TruncationSize>
                            </AirSyncBase:BodyPreference>
                            <Conflict>1</Conflict>
                        </Options>
                        <Commands>
                            <Change>
                                <ServerId>{$params[1]}</ServerId>
                                <ApplicationData>
                                    <Contacts:FirstName>First</Contacts:FirstName>
                                    <Contacts:LastName>Last</Contacts:LastName>
                                </ApplicationData>
                            </Change>
                        </Commands>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(0);
        $this->assertSame('1', $xpath->query("ns:Status", $root)->item(0)->nodeValue);
        $this->assertSame(strval(++$params[0]), $xpath->query("ns:SyncKey", $root)->item(0)->nodeValue);
        $this->assertSame(0, $xpath->query("ns:Responses", $root)->length);

        // TODO: Assert updated content on the server

        return $params;
    }

    /**
     * Test deleting objects from client
     *
     * @depends testChangeFromClient
     */
    public function testDeleteFromClient($params)
    {
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase" xmlns:Contacts="uri:Contacts">
                <Collections>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>{$params[0]}</SyncKey>
                        <CollectionId>Contacts::Syncroton</CollectionId>
                        <DeletesAsMoves/>
                        <GetChanges/>
                        <Options>
                            <AirSyncBase:BodyPreference>
                                <AirSyncBase:Type>1</AirSyncBase:Type>
                                <AirSyncBase:TruncationSize>5120</AirSyncBase:TruncationSize>
                            </AirSyncBase:BodyPreference>
                            <Conflict>1</Conflict>
                        </Options>
                        <Commands>
                            <Delete>
                                <ServerId>{$params[1]}</ServerId>
                            </Delete>
                        </Commands>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(0);
        $this->assertSame('1', $xpath->query("ns:Status", $root)->item(0)->nodeValue);
        $this->assertSame(strval(++$params[0]), $xpath->query("ns:SyncKey", $root)->item(0)->nodeValue);
        $this->assertSame(0, $xpath->query("ns:Responses", $root)->length);

        // TODO: Assert deleted contact on the server
    }
}
