<?php

/**
 * Kolab Tags engine
 *
 * @author Aleksander Machniak <machniak@kolabsys.com>
 *
 * Copyright (C) 2014, Kolab Systems AG <contact@kolabsys.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

class kolab_tags_engine
{
    private $backend;
    private $plugin;
    private $rc;

    /**
     * Class constructor
     */
    public function __construct($plugin)
    {
        $plugin->require_plugin('libkolab');

        $this->plugin  = $plugin;
        $this->rc      = $plugin->rc;

        $driver = $this->rc->config->get('kolab_tags_driver') ?: 'database';
        $class = "\\KolabTags\\Drivers\\" . ucfirst($driver) . "\\Driver";

        require_once "{$plugin->home}/drivers/DriverInterface.php";
        require_once "{$plugin->home}/drivers/{$driver}/Driver.php";

        $this->backend = new $class();
    }

    /**
     * User interface initialization
     */
    public function ui()
    {
        if ($this->rc->action && !in_array($this->rc->action, ['show', 'preview', 'dialog-ui'])) {
            return;
        }

        $this->plugin->add_texts('localization/');

        $this->plugin->include_stylesheet($this->plugin->local_skin_path() . '/style.css');
        $this->plugin->include_script('kolab_tags.js');
        $this->rc->output->add_label('cancel', 'save');
        $this->plugin->add_label(
            'tags',
            'add',
            'edit',
            'delete',
            'saving',
            'nameempty',
            'nameexists',
            'colorinvalid',
            'untag',
            'tagname',
            'tagcolor',
            'tagsearchnew',
            'newtag',
            'notags'
        );

        $this->rc->output->add_handlers([
            'plugin.taglist' => [$this, 'taglist'],
        ]);

        $ui = $this->rc->output->parse('kolab_tags.ui', false, false);
        $this->rc->output->add_footer($ui);

        // load miniColors and tagedit
        jqueryui::miniColors();
        jqueryui::tagedit();

        // Modify search filter (and set selected tags)
        if ($this->rc->task == 'mail' && ($this->rc->action == 'show' || !$this->rc->action)) {
            $this->search_filter_mods();
        }
    }

    /**
     * Engine actions handler (managing tag objects)
     */
    public function actions()
    {
        $this->plugin->add_texts('localization/');

        $action = rcube_utils::get_input_value('_act', rcube_utils::INPUT_POST);

        if ($action) {
            $this->{'action_' . $action}();
        }
        // manage tag objects
        else {
            $delete   = (array) rcube_utils::get_input_value('delete', rcube_utils::INPUT_POST);
            $update   = (array) rcube_utils::get_input_value('update', rcube_utils::INPUT_POST, true);
            $add      = (array) rcube_utils::get_input_value('add', rcube_utils::INPUT_POST, true);
            $response = [];

            // tags deletion
            foreach ($delete as $uid) {
                if ($this->backend->remove($uid)) {
                    $response['delete'][] = $uid;
                } else {
                    $error = true;
                }
            }

            // tags creation
            foreach ($add as $tag) {
                if ($tag = $this->backend->create($tag)) {
                    $response['add'][] = $this->parse_tag($tag);
                } else {
                    $error = true;
                }
            }

            // tags update
            foreach ($update as $tag) {
                if ($this->backend->update($tag)) {
                    $response['update'][] = $this->parse_tag($tag);
                } else {
                    $error = true;
                }
            }

            if (!empty($error)) {
                $this->rc->output->show_message($this->plugin->gettext('updateerror'), 'error');
            } else {
                $this->rc->output->show_message($this->plugin->gettext('updatesuccess'), 'confirmation');
            }

            $this->rc->output->command('plugin.kolab_tags', $response);
        }

        $this->rc->output->send();
    }

    /**
     * Remove tag from message(s)
     */
    public function action_remove()
    {
        $tag = rcube_utils::get_input_value('_tag', rcube_utils::INPUT_POST);
        $filter = $tag == '*' ? [] : [['uid', '=', explode(',', $tag)]];
        $taglist = $this->backend->list_tags($filter);
        $tags = [];

        // for every tag...
        foreach ($taglist as $tag) {
            $error = !$this->backend->remove_tag_members($tag, rcmail::get_uids());

            if ($error) {
                break;
            }

            $tags[] = $tag['uid'];
        }

        if (!empty($error)) {
            if (!isset($_POST['_from']) || $_POST['_from'] != 'show') {
                $this->rc->output->show_message($this->plugin->gettext('untaggingerror'), 'error');
                $this->rc->output->command('list_mailbox');
            }
        } else {
            $this->rc->output->show_message($this->plugin->gettext('untaggingsuccess'), 'confirmation');
            $this->rc->output->command('plugin.kolab_tags', ['mark' => 1, 'delete' => $tags]);
        }
    }

    /**
     * Add tag to message(s)
     */
    public function action_add()
    {
        $tag = rcube_utils::get_input_value('_tag', rcube_utils::INPUT_POST);
        $taglist = [];

        // create a new tag?
        if (!empty($_POST['_new'])) {
            $object = [
                'name' => $tag,
            ];

            $object = $this->backend->create($object);
            $error  = $object === false;
            if ($object) {
                $taglist[] = $object;
            }
        }
        // use existing tags (by UID)
        else {
            $filter  = [['uid', '=', explode(',', $tag)]];
            $taglist = $this->backend->list_tags($filter);
        }

        if (empty($error)) {
            // for every tag...
            foreach ($taglist as $tag) {
                $error = !$this->backend->add_tag_members($tag, rcmail::get_uids());
                if ($error) {
                    break;
                }
            }
        }

        if (!empty($error)) {
            $this->rc->output->show_message($this->plugin->gettext('taggingerror'), 'error');

            if ($_POST['_from'] != 'show') {
                $this->rc->output->command('list_mailbox');
            }
        } else {
            $this->rc->output->show_message($this->plugin->gettext('taggingsuccess'), 'confirmation');

            if (isset($object)) {
                $this->rc->output->command('plugin.kolab_tags', ['mark' => 1, 'add' => [$this->parse_tag($object)]]);
            }
        }
    }

    /**
     * Refresh tags list
     */
    public function action_refresh()
    {
        $taglist = $this->backend->list_tags();
        $taglist = array_map([$this, 'parse_tag'], $taglist);

        $this->rc->output->set_env('tags', $taglist);
        $this->rc->output->command('plugin.kolab_tags', ['refresh' => 1]);
    }

    /**
     * Template object building tags list/cloud
     */
    public function taglist($attrib)
    {
        $taglist = $this->backend->list_tags();
        $taglist = array_map([$this, 'parse_tag'], $taglist);

        $this->rc->output->set_env('tags', $taglist);
        $this->rc->output->add_gui_object('taglist', $attrib['id']);

        return html::tag('ul', $attrib, '', html::$common_attrib);
    }

    /**
     * Handler for messages list (add tag-boxes in subject line on the list)
     */
    public function messages_list_handler($args)
    {
        if (empty($args['messages'])) {
            return;
        }

        $this->rc->output->set_env('message_tags', $this->backend->members_tags($args['messages']));

        // @TODO: tag counters for the whole folder (search result)

        return $args;
    }

    /**
     * Handler for a single message (add tag-boxes in subject line)
     */
    public function message_headers_handler($args)
    {
        $taglist = $this->backend->list_tags();

        if (!empty($taglist)) {
            $tag_uids = $this->backend->members_tags([$args['headers']]);

            if (!empty($tag_uids)) {
                $tag_uids = array_first($tag_uids);
                $taglist = array_filter($taglist, function ($tag) use ($tag_uids) {
                    return in_array($tag['uid'], $tag_uids);
                });
                $taglist = array_map([$this, 'parse_tag'], $taglist);

                $this->rc->output->set_env('message_tags', $taglist);
            }
        }

        return $args;
    }

    /**
     * Handler for messages searching requests
     */
    public function imap_search_handler($args)
    {
        if (empty($args['search_tags'])) {
            return $args;
        }

        // we'll reset to current folder to fix issues when searching in multi-folder mode
        $storage     = $this->rc->get_storage();
        $orig_folder = $storage->get_folder();

        // get tags
        $tags = $this->backend->list_tags([['uid', '=', $args['search_tags']]]);

        // sanity check (that should not happen)
        // TODO: This is driver-specific and should be moved to drivers
        if (empty($tags)) {
            if ($orig_folder) {
                $storage->set_folder($orig_folder);
            }

            return $args;
        }

        $criteria = [];
        $folders = $args['folder'] = (array) $args['folder'];
        $search = $args['search'];

        foreach ($tags as $tag) {
            $res = $this->backend->members_search_criteria($tag, $args['folder']);

            if (empty($res)) {
                // If any tag has no members in any folder we can skip the other tags
                goto empty_result;
            }

            $criteria = array_intersect_key($criteria, $res);
            $args['folder'] = array_keys($res);

            foreach ($res as $folder => $value) {
                $current = !empty($criteria[$folder]) ? $criteria[$folder] : trim($search);
                $criteria[$folder] = ($current == 'ALL' ? '' : ($current . ' ')) . $value;
            }
        }

        if (!empty($args['folder'])) {
            $args['search'] = $criteria;
        } else {
            // return empty result
            empty_result:

            if (count($folders) > 1) {
                $args['result'] = new rcube_result_multifolder($folders);
                foreach ($folders as $folder) {
                    $index = new rcube_result_index($folder, '* SORT');
                    $args['result']->add($index);
                }
            } else {
                $class  = 'rcube_result_' . ($args['threading'] ? 'thread' : 'index');
                $result = $args['threading'] ? '* THREAD' : '* SORT';

                $args['result'] = new $class(array_first($folders) ?: 'INBOX', $result);
            }
        }

        if ($orig_folder) {
            $storage->set_folder($orig_folder);
        }

        return $args;
    }

    /**
     * Get selected tags when in search-mode
     */
    protected function search_filter_mods()
    {
        if (!empty($_REQUEST['_search']) && !empty($_SESSION['search'])
             && $_SESSION['search_request'] == $_REQUEST['_search']
             && ($filter = $_SESSION['search_filter'])
        ) {
            if (preg_match('/^(kolab_tags_[0-9]{10,}:([^:]+):)/', $filter, $m)) {
                $search_tags   = explode(',', $m[2]);
                $search_filter = substr($filter, strlen($m[1]));

                // send current search properties to the browser
                $this->rc->output->set_env('search_filter_selected', $search_filter);
                $this->rc->output->set_env('selected_tags', $search_tags);
            }
        }
    }

    /**
     * "Convert" tag object to simple array for use in javascript
     */
    private function parse_tag($tag)
    {
        return [
            'uid'   => $tag['uid'],
            'name'  => $tag['name'],
            'color' => $tag['color'] ?? null,
            'immutableName' => !empty($this->backend->immutableName),
        ];
    }
}
