<?php

/**
 * Kolab Tags backend driver for IMAP ANNOTATE (and METADATA). E.g. Kolab4.
 *
 * @author Aleksander Machniak <machniak@apheleia-it.ch>
 *
 * Copyright (C) 2024, Apheleia IT AG <contact@apheleia-it.ch>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

namespace KolabTags\Drivers\Annotate;

use KolabTags\Drivers\DriverInterface;
use kolab_storage_tags;
use rcube;
use rcube_imap_generic;
use rcube_message_header;

class Driver implements DriverInterface
{
    public $immutableName = true;

    protected $engine;

    /**
     * Class constructor
     */
    public function __construct()
    {
        $this->engine = new kolab_storage_tags();
    }

    /**
     * Tags list
     *
     * @param array $filter Search filter
     *
     * @return array List of tags
     */
    public function list_tags($filter = [])
    {
        return $this->engine->list($filter);
    }

    /**
     * Create tag object
     *
     * @param array $tag Tag data
     *
     * @return false|array Tag data on success, False on failure
     */
    public function create($tag)
    {
        $tag['uid'] = $this->engine->create($tag);

        if (empty($tag['uid'])) {
            return false;
        }

        return $tag;
    }

    /**
     * Update tag object
     *
     * @param array $tag Tag data
     *
     * @return false|array Tag data on success, False on failure
     */
    public function update($tag)
    {
        $success = $this->engine->update($tag);

        return $success ? $tag : false;
    }

    /**
     * Remove tag object
     *
     * @param string $uid Object unique identifier
     *
     * @return bool True on success, False on failure
     */
    public function remove($uid)
    {
        return $this->engine->delete($uid);
    }

    /**
     * Build IMAP SEARCH criteria for mail messages search (per-folder)
     *
     * @param array $tag     Tag data
     * @param array $folders List of folders to search in
     *
     * @return array<string> IMAP SEARCH criteria per-folder
     */
    public function members_search_criteria($tag, $folders)
    {
        $result  = [];
        foreach ($folders as $folder) {
            $result[$folder] = $this->engine->imap_search_criteria($tag['name']);
        }

        return $result;
    }

    /**
     * Returns tag assignments with multiple members
     *
     * @param array<rcube_message_header> $messages Mail messages
     *
     * @return array<string, array> Tags assigned
     */
    public function members_tags($messages)
    {
        return $this->engine->members_tags($messages);
    }

    /**
     * Add mail members to a tag
     *
     * @param array $tag      Tag object
     * @param array $messages List of messages in rcmail::get_uids() output format
     *
     * @return bool True on success, False on error
     */
    public function add_tag_members($tag, $messages)
    {
        $storage = rcube::get_instance()->get_storage();
        $members = [];

        foreach ($messages as $mbox => $uids) {
            if ($uids === '*') {
                $index = $storage->index($mbox, null, null, true);
                $uids  = $index->get();
            }

            if (empty($uids)) {
                continue;
            }

            $result = $this->engine->add_members($tag['uid'], $mbox, $uids);

            if (!$result) {
                return false;
            }
        }

        return true;
    }

    /**
     * Remove mail members from a tag
     *
     * @param array $tag      Tag object
     * @param array $messages List of messages in rcmail::get_uids() output format
     *
     * @return bool True on success, False on error
     */
    public function remove_tag_members($tag, $messages)
    {
        $storage = rcube::get_instance()->get_storage();

        foreach ($messages as $mbox => $uids) {
            if ($uids === '*') {
                $index = $storage->index($mbox, null, null, true);
                $uids  = $index->get();
            }

            if (empty($uids)) {
                continue;
            }

            $result = $this->engine->remove_members($tag['uid'], $mbox, $uids);

            if (!$result) {
                return false;
            }
        }

        return true;
    }
}
