<?php

/**
 * Syncroton
 *
 * @package     Syncroton
 * @subpackage  Command
 * @license     http://www.tine20.org/licenses/lgpl.html LGPL Version 3
 * @copyright   Copyright (c) 2009-2012 Metaways Infosystems GmbH (http://www.metaways.de)
 * @author      Lars Kneschke <l.kneschke@metaways.de>
 */

/**
 * class to handle ActiveSync Sync command
 *
 * @package     Syncroton
 * @subpackage  Command
 */
class Syncroton_Command_Sync extends Syncroton_Command_Wbxml
{
    public const STATUS_SUCCESS                                = 1;
    public const STATUS_PROTOCOL_VERSION_MISMATCH              = 2;
    public const STATUS_INVALID_SYNC_KEY                       = 3;
    public const STATUS_PROTOCOL_ERROR                         = 4;
    public const STATUS_SERVER_ERROR                           = 5;
    public const STATUS_ERROR_IN_CLIENT_SERVER_CONVERSION      = 6;
    public const STATUS_CONFLICT_MATCHING_THE_CLIENT_AND_SERVER_OBJECT = 7;
    public const STATUS_OBJECT_NOT_FOUND                       = 8;
    public const STATUS_USER_ACCOUNT_MAYBE_OUT_OF_DISK_SPACE   = 9;
    public const STATUS_ERROR_SETTING_NOTIFICATION_GUID        = 10;
    public const STATUS_DEVICE_NOT_PROVISIONED_FOR_NOTIFICATIONS = 11;
    public const STATUS_FOLDER_HIERARCHY_HAS_CHANGED           = 12;
    public const STATUS_RESEND_FULL_XML                        = 13;
    public const STATUS_WAIT_INTERVAL_OUT_OF_RANGE             = 14;
    public const STATUS_TOO_MANY_COLLECTIONS                   = 15;

    public const CONFLICT_OVERWRITE_SERVER                     = 0;
    public const CONFLICT_OVERWRITE_PIM                        = 1;

    public const MIMESUPPORT_DONT_SEND_MIME                    = 0;
    public const MIMESUPPORT_SMIME_ONLY                        = 1;
    public const MIMESUPPORT_SEND_MIME                         = 2;

    public const BODY_TYPE_PLAIN_TEXT                          = 1;
    public const BODY_TYPE_HTML                                = 2;
    public const BODY_TYPE_RTF                                 = 3;
    public const BODY_TYPE_MIME                                = 4;

    /**
     * truncate types
     */
    public const TRUNCATE_ALL                                  = 0;
    public const TRUNCATE_4096                                 = 1;
    public const TRUNCATE_5120                                 = 2;
    public const TRUNCATE_7168                                 = 3;
    public const TRUNCATE_10240                                = 4;
    public const TRUNCATE_20480                                = 5;
    public const TRUNCATE_51200                                = 6;
    public const TRUNCATE_102400                               = 7;
    public const TRUNCATE_NOTHING                              = 8;

    /**
     * filter types
     */
    public const FILTER_NOTHING        = 0;
    public const FILTER_1_DAY_BACK     = 1;
    public const FILTER_3_DAYS_BACK    = 2;
    public const FILTER_1_WEEK_BACK    = 3;
    public const FILTER_2_WEEKS_BACK   = 4;
    public const FILTER_1_MONTH_BACK   = 5;
    public const FILTER_3_MONTHS_BACK  = 6;
    public const FILTER_6_MONTHS_BACK  = 7;
    public const FILTER_INCOMPLETE     = 8;


    protected $_defaultNameSpace    = 'uri:AirSync';
    protected $_documentElement     = 'Sync';

    /**
     * list of collections
     *
     * @var array<string,Syncroton_Model_SyncCollection>
     */
    protected $_collections = [];

    protected $_modifications = [];

    /**
     * the global WindowSize
     *
     * @var integer
     */
    protected $_globalWindowSize;

    /**
     * there are more entries than WindowSize available
     * the MoreAvailable tag hot added to the xml output
     *
     * @var boolean
     */
    protected $_moreAvailable = false;

    protected $_maxWindowSize = 100;

    protected $_heartbeatInterval = null;

    /**
     * process the XML file and add, change, delete or fetches data
     */
    public function handle()
    {
        // input xml
        $requestXML = simplexml_import_dom($this->_mergeSyncRequest($this->_requestBody, $this->_device));

        if (! isset($requestXML->Collections)) {
            $this->_outputDom->documentElement->appendChild(
                $this->_outputDom->createElementNS('uri:AirSync', 'Status', self::STATUS_RESEND_FULL_XML)
            );

            return $this->_outputDom;
        }

        $intervalDiv = 1;
        if (isset($requestXML->HeartbeatInterval)) {
            $intervalDiv = 1;
            $this->_heartbeatInterval = (int)$requestXML->HeartbeatInterval;
        } elseif (isset($requestXML->Wait)) {
            $intervalDiv = 60;
            $this->_heartbeatInterval = (int)$requestXML->Wait * $intervalDiv;
        }

        $maxInterval = Syncroton_Registry::getPingInterval();
        if ($maxInterval <= 0 || $maxInterval > Syncroton_Server::MAX_HEARTBEAT_INTERVAL) {
            $maxInterval = Syncroton_Server::MAX_HEARTBEAT_INTERVAL;
        }

        if ($this->_heartbeatInterval && $this->_heartbeatInterval > $maxInterval) {
            $sync = $this->_outputDom->documentElement;
            $sync->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', self::STATUS_WAIT_INTERVAL_OUT_OF_RANGE));
            $sync->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Limit', floor($maxInterval / $intervalDiv)));
            $this->_heartbeatInterval = null;
            return;
        }

        $this->_globalWindowSize = isset($requestXML->WindowSize) ? (int)$requestXML->WindowSize : 100;

        if (!$this->_globalWindowSize || $this->_globalWindowSize > 512) {
            $this->_globalWindowSize = 512;
        }

        if ($this->_globalWindowSize > $this->_maxWindowSize) {
            $this->_globalWindowSize = $this->_maxWindowSize;
        }

        // load options from lastsynccollection
        $lastSyncCollection = ['options' => []];
        if (!empty($this->_device->lastsynccollection)) {
            $lastSyncCollection = Zend_Json::decode($this->_device->lastsynccollection);
            if (!array_key_exists('options', $lastSyncCollection) || !is_array($lastSyncCollection['options'])) {
                $lastSyncCollection['options'] = [];
            }
        }

        $maxCollections = Syncroton_Registry::getMaxCollections();
        if ($maxCollections && count($requestXML->Collections->Collection) > $maxCollections) {
            $sync = $this->_outputDom->documentElement;
            $sync->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', self::STATUS_TOO_MANY_COLLECTIONS));
            $sync->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Limit', $maxCollections));
            return;
        }

        $collections = [];

        foreach ($requestXML->Collections->Collection as $xmlCollection) {
            $collectionId = (string)$xmlCollection->CollectionId;

            $collections[$collectionId] = new Syncroton_Model_SyncCollection($xmlCollection);

            // do we have to reuse the options from the previous request?
            if (!isset($xmlCollection->Options) && array_key_exists($collectionId, $lastSyncCollection['options'])) {
                $collections[$collectionId]->options = $lastSyncCollection['options'][$collectionId];
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " restored options to " . print_r($collections[$collectionId]->options, true));
                }
            }

            // store current options for next Sync command request (sticky options)
            $lastSyncCollection['options'][$collectionId] = $collections[$collectionId]->options;
        }

        $this->_device->lastsynccollection = Zend_Json::encode($lastSyncCollection);

        if ($this->_device->isDirty()) {
            Syncroton_Registry::getDeviceBackend()->update($this->_device);
        }

        foreach ($collections as $collectionData) {
            // has the folder been synchronised to the device already
            try {
                $collectionData->folder = $this->_folderBackend->getFolder($this->_device, $collectionData->collectionId);

            } catch (Syncroton_Exception_NotFound $senf) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " folder {$collectionData->collectionId} not found");
                }

                // trigger INVALID_SYNCKEY instead of OBJECT_NOTFOUND when synckey is higher than 0
                // to avoid a syncloop for the iPhone
                if ($collectionData->syncKey > 0) {
                    $collectionData->folder    = new Syncroton_Model_Folder([
                        'deviceId' => $this->_device,
                        'serverId' => $collectionData->collectionId,
                    ]);
                }

                $this->_collections[$collectionData->collectionId] = $collectionData;

                continue;
            }

            if ($this->_logger instanceof Zend_Log) {
                $this->_logger->info(__METHOD__ . '::' . __LINE__ . " SyncKey is {$collectionData->syncKey} Class: {$collectionData->folder->class} CollectionId: {$collectionData->collectionId}");
            }

            // initial synckey
            if ($collectionData->syncKey === 0) {
                // reset sync state for this folder
                $this->_syncStateBackend->resetState($this->_device, $collectionData->folder);
                $this->_contentStateBackend->resetState($this->_device, $collectionData->folder);

                $collectionData->syncState    = new Syncroton_Model_SyncState([
                    'device_id' => $this->_device,
                    'counter'   => 0,
                    'type'      => $collectionData->folder,
                    'lastsync'  => $this->_syncTimeStamp,
                ]);

                $this->_collections[$collectionData->collectionId] = $collectionData;

                continue;
            }

            $syncKeyReused = $this->_syncStateBackend->haveNext($this->_device, $collectionData->folder, $collectionData->syncKey);
            if ($syncKeyReused) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " already known synckey {$collectionData->syncKey} provided");
                }
            }
            // check for invalid synckey
            if (($collectionData->syncState = $this->_syncStateBackend->validate($this->_device, $collectionData->folder, $collectionData->syncKey)) === false) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " invalid synckey {$collectionData->syncKey} provided");
                }

                // reset sync state for this folder
                $this->_syncStateBackend->resetState($this->_device, $collectionData->folder);
                $this->_contentStateBackend->resetState($this->_device, $collectionData->folder);

                $this->_collections[$collectionData->collectionId] = $collectionData;

                continue;
            }

            $dataController = Syncroton_Data_Factory::factory($collectionData->folder->class, $this->_device, $this->_syncTimeStamp);

            switch ($collectionData->folder->class) {
                case Syncroton_Data_Factory::CLASS_CALENDAR:
                    $dataClass = 'Syncroton_Model_Event';
                    break;

                case Syncroton_Data_Factory::CLASS_CONTACTS:
                    $dataClass = 'Syncroton_Model_Contact';
                    break;

                case Syncroton_Data_Factory::CLASS_EMAIL:
                    $dataClass = 'Syncroton_Model_Email';
                    break;

                case Syncroton_Data_Factory::CLASS_NOTES:
                    $dataClass = 'Syncroton_Model_Note';
                    break;

                case Syncroton_Data_Factory::CLASS_TASKS:
                    $dataClass = 'Syncroton_Model_Task';
                    break;

                default:
                    throw new Syncroton_Exception_UnexpectedValue('invalid class provided');
            }

            $clientModifications = [
                'added'            => [],
                'changed'          => [],
                'deleted'          => [],
                'forceAdd'         => [],
                'forceChange'      => [],
                'toBeFetched'      => [],
            ];

            // handle incoming data
            if ($collectionData->hasClientAdds()) {
                $adds = $collectionData->getClientAdds();

                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " found " . count($adds) . " entries to be added to server");
                }

                $clientIdMap = [];
                if ($syncKeyReused && $collectionData->syncState->clientIdMap) {
                    $clientIdMap = Zend_Json::decode($collectionData->syncState->clientIdMap);
                }

                foreach ($adds as $add) {
                    if ($this->_logger instanceof Zend_Log) {
                        $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " add entry with clientId " . (string) $add->ClientId);
                    }

                    try {
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->info(__METHOD__ . '::' . __LINE__ . " adding entry as new");
                        }

                        $clientId = (string)$add->ClientId;
                        // If the sync key was reused, but we don't have a $clientId mapping,
                        // this means the client sent a new item with the same sync_key.
                        if ($syncKeyReused && array_key_exists($clientId, $clientIdMap)) {
                            // We don't normally store the clientId, so if a command with Add's is resent,
                            // we have to look-up the corresponding serverId using a cached clientId => serverId mapping,
                            // otherwise we would duplicate all added items on resend.
                            $serverId = $clientIdMap[$clientId];
                            $clientModifications['added'][$serverId] = [
                                'clientId'     => (string)$add->ClientId,
                                'serverId'     => $serverId,
                                'status'       => self::STATUS_SUCCESS,
                                'contentState' => null,
                            ];
                        } else {
                            $serverId = $dataController->createEntry($collectionData->collectionId, new $dataClass($add->ApplicationData));
                            $clientModifications['added'][$serverId] = [
                                'clientId'     => (string)$add->ClientId,
                                'serverId'     => $serverId,
                                'status'       => self::STATUS_SUCCESS,
                                'contentState' => $this->_contentStateBackend->create(new Syncroton_Model_Content([
                                    'device_id'        => $this->_device,
                                    'folder_id'        => $collectionData->folder,
                                    'contentid'        => $serverId,
                                    'creation_time'    => $this->_syncTimeStamp,
                                    'creation_synckey' => $collectionData->syncKey + 1,
                                ])),
                            ];
                        }

                    } catch (Exception $e) {
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " failed to add entry " . $e->getMessage());
                        }
                        $clientModifications['added'][] = [
                            'clientId' => (string)$add->ClientId,
                            'status'   => self::STATUS_SERVER_ERROR,
                        ];
                    }
                }
            }

            // handle changes, but only if not first sync
            if (!$syncKeyReused && $collectionData->syncKey > 1 && $collectionData->hasClientChanges()) {
                $changes = $collectionData->getClientChanges();

                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " found " . count($changes) . " entries to be updated on server");
                }

                foreach ($changes as $change) {
                    $serverId = (string)$change->ServerId;

                    try {
                        $dataController->updateEntry($collectionData->collectionId, $serverId, new $dataClass($change->ApplicationData));
                        $clientModifications['changed'][$serverId] = self::STATUS_SUCCESS;

                    } catch (Syncroton_Exception_AccessDenied $e) {
                        $clientModifications['changed'][$serverId] = self::STATUS_CONFLICT_MATCHING_THE_CLIENT_AND_SERVER_OBJECT;
                        $clientModifications['forceChange'][$serverId] = $serverId;

                    } catch (Syncroton_Exception_NotFound $e) {
                        // entry does not exist anymore, will get deleted automaticaly
                        $clientModifications['changed'][$serverId] = self::STATUS_OBJECT_NOT_FOUND;

                    } catch (Exception $e) {
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " failed to update entry " . $e);
                        }
                        // something went wrong while trying to update the entry
                        $clientModifications['changed'][$serverId] = self::STATUS_SERVER_ERROR;
                    }
                }
            }

            // handle deletes, but only if not first sync
            if (!$syncKeyReused && $collectionData->hasClientDeletes()) {
                $deletes = $collectionData->getClientDeletes();
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " found " . count($deletes) . " entries to be deleted on server");
                }

                foreach ($deletes as $delete) {
                    $serverId = (string)$delete->ServerId;

                    try {
                        // check if we have sent this entry to the phone
                        $state = $this->_contentStateBackend->getContentState($this->_device, $collectionData->folder, $serverId);

                        try {
                            $dataController->deleteEntry($collectionData->collectionId, $serverId, $collectionData);

                        } catch (Syncroton_Exception_NotFound $e) {
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->crit(__METHOD__ . '::' . __LINE__ . ' tried to delete entry ' . $serverId . ' but entry was not found');
                            }

                        } catch (Syncroton_Exception $e) {
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->info(__METHOD__ . '::' . __LINE__ . ' tried to delete entry ' . $serverId . ' but a error occured: ' . $e->getMessage());
                            }
                            $clientModifications['forceAdd'][$serverId] = $serverId;
                        }
                        $this->_contentStateBackend->delete($state);

                    } catch (Syncroton_Exception_NotFound $senf) {
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->info(__METHOD__ . '::' . __LINE__ . ' ' . $serverId . ' should have been removed from client already');
                        }
                        // should we send a special status???
                        //$collectionData->deleted[$serverId] = self::STATUS_SUCCESS;
                    }

                    $clientModifications['deleted'][$serverId] = self::STATUS_SUCCESS;
                }
            }

            // handle fetches, but only if not first sync
            if ($collectionData->syncKey > 1 && $collectionData->hasClientFetches()) {
                // the default value for GetChanges is 1. If the phone don't want the changes it must set GetChanges to 0
                // some prevoius versions of iOS did not set GetChanges to 0 for fetches. Let's enforce getChanges to false here.
                $collectionData->getChanges = false;

                $fetches = $collectionData->getClientFetches();
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " found " . count($fetches) . " entries to be fetched from server");
                }

                $toBeFetched = [];

                foreach ($fetches as $fetch) {
                    $serverId = (string)$fetch->ServerId;

                    $toBeFetched[$serverId] = $serverId;
                }

                $collectionData->toBeFetched = $toBeFetched;
            }

            $this->_collections[$collectionData->collectionId] = $collectionData;
            $this->_modifications[$collectionData->collectionId] = $clientModifications;
        }
    }

    private function getServerModifications($dataController, $collectionData, $clientModifications)
    {
        $serverModifications = [
            'added'   => [],
            'changed' => [],
            'deleted' => [],
        ];

        // We first use hasChanges because it has a fast path for when there are no changes by fetching the count of messages only.
        // However, in all other cases we will end up fetching the same entries as below, which is less than ideal.
        // TODO: We should create a new method, which checks if there are no changes, and otherwise just let the code below figure out
        // if there are any changes to process.
        if (!$dataController->hasChanges($this->_contentStateBackend, $collectionData->folder, $collectionData->syncState)) {
            return $serverModifications;
        }

        // update _syncTimeStamp as $dataController->hasChanges might have spent some time
        $this->_syncTimeStamp = new DateTime('now', new DateTimeZone('UTC'));

        // fetch entries added since last sync
        $allClientEntries = $this->_contentStateBackend->getFolderState(
            $this->_device,
            $collectionData->folder,
            $collectionData->syncState->counter
        );

        // fetch entries changed since last sync
        $allChangedEntries = $dataController->getChangedEntries(
            $collectionData->collectionId,
            $collectionData->syncState,
            $collectionData->options['filterType']
        );

        // fetch all entries
        $allServerEntries = $dataController->getServerEntries(
            $collectionData->collectionId,
            $collectionData->options['filterType']
        );

        // add entries
        $serverDiff = array_diff($allServerEntries, $allClientEntries);
        // add entries which produced problems during delete from client
        $serverModifications['added'] = $clientModifications['forceAdd'];
        // add entries not yet sent to client
        $serverModifications['added'] = array_unique(array_merge($serverModifications['added'], $serverDiff));

        // @todo still needed?
        foreach ($serverModifications['added'] as $id => $serverId) {
            // skip entries added by client during this sync session
            if (isset($clientModifications['added'][$serverId]) && !isset($clientModifications['forceAdd'][$serverId])) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " skipped added entry: " . $serverId);
                }
                unset($serverModifications['added'][$id]);
            }
        }

        // entries to be deleted
        $serverModifications['deleted'] = array_diff($allClientEntries, $allServerEntries);

        // entries changed since last sync
        $serverModifications['changed'] = array_merge($allChangedEntries, $clientModifications['forceChange']);

        foreach ($serverModifications['changed'] as $id => $serverId) {
            // skip entry, if it got changed by client during current sync
            if (isset($clientModifications['changed'][$serverId]) && !isset($clientModifications['forceChange'][$serverId])) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " skipped changed entry: " . $serverId);
                }
                unset($serverModifications['changed'][$id]);
            }
            // skip entry, make sure we don't sent entries already added by client in this request
            elseif (isset($clientModifications['added'][$serverId]) && !isset($clientModifications['forceAdd'][$serverId])) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " skipped change for added entry: " . $serverId);
                }
                unset($serverModifications['changed'][$id]);
            }
        }

        // entries comeing in scope are already in $serverModifications['added'] and do not need to
        // be send with $serverCanges
        $serverModifications['changed'] = array_diff($serverModifications['changed'], $serverModifications['added']);

        return $serverModifications;
    }

    /**
     * (non-PHPdoc)
     * @see Syncroton_Command_Wbxml::getResponse()
     */
    public function getResponse()
    {
        $sync = $this->_outputDom->documentElement;

        $collections = $this->_outputDom->createElementNS('uri:AirSync', 'Collections');

        $totalChanges = 0;

        // Detect devices that do not support empty Sync reponse
        $emptySyncSupported = !preg_match('/(meego|nokian800)/i', $this->_device->useragent);

        // continue only if there are changes or no time is left
        if ($this->_heartbeatInterval > 0) {
            $intervalStart  = time();
            $sleepCallback  = Syncroton_Registry::getSleepCallback();
            $wakeupCallback = Syncroton_Registry::getWakeupCallback();

            do {
                // take a break to save battery lifetime
                $sleepCallback();
                sleep(Syncroton_Registry::getPingTimeout());

                // make sure the connection is still alive, abort otherwise
                if (connection_aborted()) {
                    if ($this->_logger instanceof Zend_Log) {
                        $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " Exiting on aborted connection");
                    }
                    exit;
                }

                $wakeupCallback();

                $now = new DateTime('now', new DateTimeZone('UTC'));

                foreach ($this->_collections as $collectionData) {
                    // continue immediately if folder does not exist
                    if (! ($collectionData->folder instanceof Syncroton_Model_IFolder)) {
                        break 2;

                        // countinue immediately if syncstate is invalid
                    } elseif (! ($collectionData->syncState instanceof Syncroton_Model_ISyncState)) {
                        break 2;

                    } else {
                        if ($collectionData->getChanges !== true) {
                            continue;
                        }

                        try {
                            // just check if the folder still exists
                            $this->_folderBackend->get($collectionData->folder);
                        } catch (Syncroton_Exception_NotFound $senf) {
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " collection does not exist anymore: " . $collectionData->collectionId);
                            }

                            $collectionData->getChanges = false;

                            // make sure this is the last while loop
                            // no break 2 here, as we like to check the other folders too
                            $intervalStart -= $this->_heartbeatInterval;
                        }

                        // check that the syncstate still exists and is still valid
                        try {
                            $syncState = $this->_syncStateBackend->getSyncState($this->_device, $collectionData->folder);

                            // another process synchronized data of this folder already. let's skip it
                            if ($syncState->id !== $collectionData->syncState->id) {
                                if ($this->_logger instanceof Zend_Log) {
                                    $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " syncstate changed during heartbeat interval for collection: " . $collectionData->folder->serverId);
                                }

                                $collectionData->getChanges = false;

                                // make sure this is the last while loop
                                // no break 2 here, as we like to check the other folders too
                                $intervalStart -= $this->_heartbeatInterval;
                            }
                        } catch (Syncroton_Exception_NotFound $senf) {
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " no syncstate found anymore for collection: " . $collectionData->folder->serverId);
                            }

                            $collectionData->syncState = null;

                            // make sure this is the last while loop
                            // no break 2 here, as we like to check the other folders too
                            $intervalStart -= $this->_heartbeatInterval;
                        }


                        // safe battery time by skipping folders which got synchronied less than Syncroton_Command_Ping::$quietTime seconds ago
                        if (! $collectionData->syncState instanceof Syncroton_Model_SyncState ||
                             ($now->getTimestamp() - $collectionData->syncState->lastsync->getTimestamp()) < Syncroton_Registry::getQuietTime()) {
                            continue;
                        }

                        $dataController = Syncroton_Data_Factory::factory($collectionData->folder->class, $this->_device, $this->_syncTimeStamp);

                        // countinue immediately if there are any changes available
                        if ($dataController->hasChanges($this->_contentStateBackend, $collectionData->folder, $collectionData->syncState)) {
                            break 2;
                        }
                    }
                }

                // See: http://www.tine20.org/forum/viewtopic.php?f=12&t=12146
                //
                // break if there are less than PingTimeout + 10 seconds left for the next loop
                // otherwise the response will be returned after the client has finished his Ping
                // request already maybe
            } while (Syncroton_Server::validateSession() && time() - $intervalStart < $this->_heartbeatInterval - (Syncroton_Registry::getPingTimeout() + 10));
        }

        // First check for folders hierarchy changes
        foreach ($this->_collections as $collectionData) {
            if (! ($collectionData->folder instanceof Syncroton_Model_IFolder)) {

                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " Detected a folder hierarchy change on {$collectionData->collectionId}.");
                }

                $sync->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', self::STATUS_FOLDER_HIERARCHY_HAS_CHANGED));
                return $this->_outputDom;
            }
        }

        // Handle Calendar folders first.
        // Outlook can't match MeetingRequest with an event that wasn't synced yet, which
        // leads to duplicated events in Outlook's calendar.
        uasort(
            $this->_collections,
            function ($a, $b) {
                $ac = $a->folder && $a->folder->class == Syncroton_Data_Factory::CLASS_CALENDAR;
                $bc = $b->folder && $b->folder->class == Syncroton_Data_Factory::CLASS_CALENDAR;
                if ($ac) {
                    return $bc ? 0 : -1;
                }
                return $bc ? 1 : 0;
            }
        );

        foreach ($this->_collections as $collectionData) {
            if ($this->_logger instanceof Zend_Log) {
                $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " Processing {$collectionData->collectionId}...");
            }

            $collectionChanges = 0;

            /**
             * keep track of entries added on server side
             */
            $newContentStates = [];

            /**
             * keep track of entries deleted on server side
             */
            $deletedContentStates = [];

            // invalid synckey provided
            if (! ($collectionData->syncState instanceof Syncroton_Model_ISyncState)) {
                // set synckey to 0
                $collection = $collections->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Collection'));
                $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'SyncKey', 0));
                $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'CollectionId', $collectionData->collectionId));
                $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', self::STATUS_INVALID_SYNC_KEY));

                // initial sync
            } elseif ($collectionData->syncState->counter === 0) {
                $collectionData->syncState->counter++;

                // initial sync
                // send back a new SyncKey only
                $collection = $collections->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Collection'));
                if (!empty($collectionData->folder->class)) {
                    $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Class', $collectionData->folder->class));
                }
                $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'SyncKey', $collectionData->syncState->counter));
                $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'CollectionId', $collectionData->collectionId));
                $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', self::STATUS_SUCCESS));
            } else {
                $dataController = Syncroton_Data_Factory::factory($collectionData->folder->class, $this->_device, $this->_syncTimeStamp);

                $clientModifications = $this->_modifications[$collectionData->collectionId];
                $serverModifications = [
                    'added'   => [],
                    'changed' => [],
                    'deleted' => [],
                ];

                $status = self::STATUS_SUCCESS;

                if ($collectionData->getChanges === true) {
                    // continue sync session?
                    if (is_array($collectionData->syncState->pendingdata)) {
                        $serverModifications = $collectionData->syncState->pendingdata;
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->info(__METHOD__ . '::' . __LINE__ . " restored from sync state.");
                        }
                    } else {
                        try {
                            $serverModifications = $this->getServerModifications($dataController, $collectionData, $clientModifications);
                        } catch (Syncroton_Exception_NotFound $e) {
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " Folder changes checking failed (not found): " . $e->getTraceAsString());
                            }

                            $status = self::STATUS_FOLDER_HIERARCHY_HAS_CHANGED;
                        } catch (Exception $e) {
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->crit(__METHOD__ . '::' . __LINE__ . " Folder changes checking failed: " . $e->getMessage());
                            }

                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " Folder state checking failed: " . $e->getTraceAsString());
                            }

                            // Prevent from removing client entries when getServerEntries() fails
                            // @todo: should we break the loop here?
                            $status = self::STATUS_SERVER_ERROR;
                        }
                    }

                    if ($this->_logger instanceof Zend_Log) {
                        $this->_logger->info(__METHOD__ . '::' . __LINE__ . " found (added/changed/deleted) " . count($serverModifications['added']) . '/' . count($serverModifications['changed']) . '/' . count($serverModifications['deleted']) . ' entries for sync from server to client');
                    }
                }

                // collection header
                $collection = $this->_outputDom->createElementNS('uri:AirSync', 'Collection');
                if (!empty($collectionData->folder->class)) {
                    $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Class', $collectionData->folder->class));
                }

                $syncKeyElement = $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'SyncKey'));

                $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'CollectionId', $collectionData->collectionId));
                $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', $status));

                $responses = $this->_outputDom->createElementNS('uri:AirSync', 'Responses');

                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " Processing collection " . $collectionData->collectionId);
                }

                // send reponse for newly added entries
                if (!empty($clientModifications['added'])) {
                    foreach ($clientModifications['added'] as $entryData) {
                        $add = $responses->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Add'));
                        $add->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ClientId', $entryData['clientId']));
                        // we have no serverId if the add failed
                        if (isset($entryData['serverId'])) {
                            $add->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ServerId', $entryData['serverId']));
                        }
                        $add->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', $entryData['status']));
                    }
                }

                // send reponse for changed entries
                if (!empty($clientModifications['changed'])) {
                    foreach ($clientModifications['changed'] as $serverId => $status) {
                        if ($status !== Syncroton_Command_Sync::STATUS_SUCCESS) {
                            $change = $responses->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Change'));
                            $change->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ServerId', $serverId));
                            $change->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', $status));
                        }
                    }
                }

                // send response for to be fetched entries
                if (!empty($collectionData->toBeFetched)) {
                    // unset all truncation settings as entries are not allowed to be truncated during fetch
                    $fetchCollectionData = clone $collectionData;

                    // unset truncationSize
                    if (isset($fetchCollectionData->options['bodyPreferences']) && is_array($fetchCollectionData->options['bodyPreferences'])) {
                        foreach ($fetchCollectionData->options['bodyPreferences'] as $key => $bodyPreference) {
                            unset($fetchCollectionData->options['bodyPreferences'][$key]['truncationSize']);
                        }
                    }
                    $fetchCollectionData->options['mimeTruncation'] = Syncroton_Command_Sync::TRUNCATE_NOTHING;

                    foreach ($collectionData->toBeFetched as $serverId) {
                        $fetch = $responses->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Fetch'));
                        $fetch->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ServerId', $serverId));

                        try {
                            $applicationData = $this->_outputDom->createElementNS('uri:AirSync', 'ApplicationData');

                            $dataController
                                ->getEntry($fetchCollectionData, $serverId)
                                ->appendXML($applicationData, $this->_device);

                            $fetch->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', self::STATUS_SUCCESS));

                            $fetch->appendChild($applicationData);
                        } catch (Exception $e) {
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " unable to convert entry to xml: " . $e->getMessage());
                            }
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " unable to convert entry to xml: " . $e->getTraceAsString());
                            }
                            $fetch->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'Status', self::STATUS_OBJECT_NOT_FOUND));
                        }
                    }
                }

                if ($responses->hasChildNodes() === true) {
                    $collection->appendChild($responses);
                }

                $commands = $this->_outputDom->createElementNS('uri:AirSync', 'Commands');

                foreach ($serverModifications['added'] as $id => $serverId) {
                    if ($collectionChanges == $collectionData->windowSize || $totalChanges + $collectionChanges >= $this->_globalWindowSize) {
                        break;
                    }

                    try {
                        $add = $this->_outputDom->createElementNS('uri:AirSync', 'Add');
                        $add->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ServerId', $serverId));

                        $applicationData = $add->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ApplicationData'));

                        $dataController
                            ->getEntry($collectionData, $serverId)
                            ->appendXML($applicationData, $this->_device);

                        $commands->appendChild($add);

                        $newContentStates[] = new Syncroton_Model_Content([
                            'device_id'        => $this->_device,
                            'folder_id'        => $collectionData->folder,
                            'contentid'        => $serverId,
                            'creation_time'    => $this->_syncTimeStamp,
                            'creation_synckey' => $collectionData->syncState->counter + 1,
                        ]);

                        $collectionChanges++;
                    } catch (Syncroton_Exception_MemoryExhausted $seme) {
                        // continue to next entry, as there is not enough memory left for the current entry
                        // this will lead to MoreAvailable at the end and the entry will be synced during the next Sync command
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->info(__METHOD__ . '::' . __LINE__ . " memory exhausted for entry: " . $serverId);
                        }

                        break;

                    } catch (Exception $e) {
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " unable to convert entry to xml: " . $e->getMessage());
                        }
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " unable to convert entry to xml: " . $e->getTraceAsString());
                        }
                        // We bump collectionChanges anyways to make sure the windowSize still applies.
                        $collectionChanges++;
                    }

                    // mark as sent to the client, even the conversion to xml might have failed
                    unset($serverModifications['added'][$id]);
                }

                /**
                 * process entries changed on server side
                 */
                foreach ($serverModifications['changed'] as $id => $serverId) {
                    if ($collectionChanges == $collectionData->windowSize || $totalChanges + $collectionChanges >= $this->_globalWindowSize) {
                        break;
                    }

                    try {
                        $change = $this->_outputDom->createElementNS('uri:AirSync', 'Change');
                        $change->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ServerId', $serverId));

                        $applicationData = $change->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ApplicationData'));

                        $collectionData->options['_changesOnly'] = true;

                        $dataController
                            ->getEntry($collectionData, $serverId)
                            ->appendXML($applicationData, $this->_device);


                        $commands->appendChild($change);

                        $collectionChanges++;
                    } catch (Syncroton_Exception_MemoryExhausted $seme) {
                        // continue to next entry, as there is not enough memory left for the current entry
                        // this will lead to MoreAvailable at the end and the entry will be synced during the next Sync command
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->info(__METHOD__ . '::' . __LINE__ . " memory exhausted for entry: " . $serverId);
                        }

                        break;

                    } catch (Exception $e) {
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " unable to convert entry to xml: " . $e->getMessage());
                        }
                        // We bump collectionChanges anyways to make sure the windowSize still applies.
                        $collectionChanges++;
                    }

                    unset($serverModifications['changed'][$id]);
                }

                foreach ($serverModifications['deleted'] as $id => $serverId) {
                    if ($collectionChanges == $collectionData->windowSize || $totalChanges + $collectionChanges >= $this->_globalWindowSize) {
                        break;
                    }

                    try {
                        // check if we have sent this entry to the phone
                        $state = $this->_contentStateBackend->getContentState($this->_device, $collectionData->folder, $serverId);

                        $delete = $this->_outputDom->createElementNS('uri:AirSync', 'Delete');
                        $delete->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'ServerId', $serverId));

                        $deletedContentStates[] = $state;

                        $commands->appendChild($delete);

                        $collectionChanges++;
                    } catch (Exception $e) {
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " unable to convert entry to xml: " . $e->getMessage());
                        }
                        // We bump collectionChanges anyways to make sure the windowSize still applies.
                        $collectionChanges++;
                    }

                    unset($serverModifications['deleted'][$id]);
                }

                $countOfPendingChanges = (count($serverModifications['added']) + count($serverModifications['changed']) + count($serverModifications['deleted']));
                if ($countOfPendingChanges > 0) {
                    if ($this->_logger instanceof Zend_Log) {
                        $this->_logger->info(__METHOD__ . '::' . __LINE__ . " there are  " . $countOfPendingChanges . " more items available");
                    }
                    $collection->appendChild($this->_outputDom->createElementNS('uri:AirSync', 'MoreAvailable'));
                } else {
                    if ($this->_logger instanceof Zend_Log) {
                        $this->_logger->info(__METHOD__ . '::' . __LINE__ . " there are no more items available");
                    }
                    $serverModifications = null;
                }

                if ($commands->hasChildNodes() === true) {
                    $collection->appendChild($commands);
                }

                $totalChanges += $collectionChanges;

                // If the client resent an old sync-key, we should still respond with the latest sync key
                if (isset($collectionData->syncState->counterNext)) {
                    //TODO we're not resending the changes in between, but I'm not sure we have to.
                    $collectionData->syncState->counter = $collectionData->syncState->counterNext;
                }

                // increase SyncKey if needed
                if ((
                    // sent the clients updates... ?
                    !empty($clientModifications['added']) ||
                    !empty($clientModifications['changed']) ||
                    !empty($clientModifications['deleted'])
                ) || (
                    // is the server sending updates to the client... ?
                    $commands->hasChildNodes() === true
                ) || (
                    // changed the pending data... ?
                    $collectionData->syncState->pendingdata != $serverModifications
                )
                ) {
                    // ...then increase SyncKey
                    $collectionData->syncState->counter++;
                }
                $syncKeyElement->appendChild($this->_outputDom->createTextNode($collectionData->syncState->counter));

                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " current synckey is " . $collectionData->syncState->counter);
                }

                if (!$emptySyncSupported || $collection->childNodes->length > 4 || $collectionData->syncState->counter != $collectionData->syncKey) {
                    $collections->appendChild($collection);
                }

                //Store next
                $collectionData->syncState->extraData = $dataController->getExtraData($collectionData->folder);
            }

            if (isset($collectionData->syncState) &&
                $collectionData->syncState instanceof Syncroton_Model_ISyncState &&
                $collectionData->syncState->counter != $collectionData->syncKey
            ) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " update syncState for collection: " . $collectionData->collectionId);
                }

                // store pending data in sync state when needed
                if (isset($countOfPendingChanges) && $countOfPendingChanges > 0) {
                    $collectionData->syncState->pendingdata = [
                        'added'   => $serverModifications['added'] ?? [],
                        'changed' => $serverModifications['changed'] ?? [],
                        'deleted' => $serverModifications['deleted'] ?? [],
                    ];
                } else {
                    $collectionData->syncState->pendingdata = null;
                }

                $collectionData->syncState->lastsync = clone $this->_syncTimeStamp;
                // increment sync timestamp by 1 second
                $collectionData->syncState->lastsync->modify('+1 sec');
                if (!empty($clientModifications['added'])) {
                    // Store a client id mapping in case we encounter a reused sync_key in a future request.
                    $newClientIdMap = [];
                    foreach ($clientModifications['added'] as $entryData) {
                        // No serverId if we failed to add
                        if ($entryData['status'] == self::STATUS_SUCCESS) {
                            $newClientIdMap[$entryData['clientId']] = $entryData['serverId'];
                        }
                    }
                    $collectionData->syncState->clientIdMap = Zend_Json::encode($newClientIdMap);
                }

                //Retry in case of deadlock
                $retryCounter = 0;
                while (true) {
                    try {
                        $transactionId = Syncroton_Registry::getTransactionManager()->startTransaction(Syncroton_Registry::getDatabase());
                        // store new synckey
                        $this->_syncStateBackend->create($collectionData->syncState, true); // @phpstan-ignore-line

                        // store contentstates for new entries added to client
                        foreach ($newContentStates as $state) {
                            try {
                                //This can happen if we rerun a previous sync-key
                                $state = $this->_contentStateBackend->getContentState($state->device_id, $state->folder_id, $state->contentid);
                                $this->_contentStateBackend->update($state);
                            } catch (Exception $zdse) {
                                $this->_contentStateBackend->create($state);
                            }
                        }

                        // remove contentstates for entries to be deleted on client
                        foreach ($deletedContentStates as $state) {
                            $this->_contentStateBackend->delete($state);
                        }

                        Syncroton_Registry::getTransactionManager()->commitTransaction($transactionId);
                        break;
                    } catch (Syncroton_Exception_DeadlockDetected $zdse) {
                        $retryCounter++;
                        if ($retryCounter > 60) {
                            if ($this->_logger instanceof Zend_Log) {
                                $this->_logger->warn(__METHOD__ . '::' . __LINE__ . ' exception while storing new synckey. Aborting after 5 retries.');
                            }

                            // something went wrong
                            // maybe another parallel request added a new synckey
                            // we must remove data added from client
                            if (!empty($clientModifications['added']) && isset($dataController)) {
                                foreach ($clientModifications['added'] as $added) {
                                    $this->_contentStateBackend->delete($added['contentState']);
                                    $dataController->deleteEntry($collectionData->collectionId, $added['serverId']);
                                }
                            }

                            Syncroton_Registry::getTransactionManager()->rollBack();

                            throw $zdse;
                        }

                        Syncroton_Registry::getTransactionManager()->rollBack();
                        // Give the other transactions some time before we try again
                        sleep(1);
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->warn(__METHOD__ . '::' . __LINE__ . ' error during transaction, trying again.');
                        }
                    } catch (Exception $zdse) {
                        if ($this->_logger instanceof Zend_Log) {
                            $this->_logger->warn(__METHOD__ . '::' . __LINE__ . ' exception while storing new synckey.');
                        }
                        // something went wrong
                        // maybe another parallel request added a new synckey
                        // we must remove data added from client
                        if (!empty($clientModifications['added']) && isset($dataController)) {
                            foreach ($clientModifications['added'] as $added) {
                                $this->_contentStateBackend->delete($added['contentState']);
                                $dataController->deleteEntry($collectionData->collectionId, $added['serverId']);
                            }
                        }

                        Syncroton_Registry::getTransactionManager()->rollBack();

                        throw $zdse;
                    }
                }
            }

            // store current filter type
            try {
                /** @var Syncroton_Model_Folder $folderState */
                $folderState = $this->_folderBackend->get($collectionData->folder);
                $folderState->lastfiltertype = $collectionData->options['filterType'];
                if ($folderState->isDirty()) {
                    $this->_folderBackend->update($folderState);
                }
            } catch (Syncroton_Exception_NotFound $senf) {
                // failed to get folderstate => should not happen but is also no problem in this state
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->warn(__METHOD__ . '::' . __LINE__ . ' failed to get folder state for: ' . $collectionData->collectionId);
                }
            }
        }

        if ($collections->hasChildNodes() === true) {
            $sync->appendChild($collections);
        }

        if ($sync->hasChildNodes()) {
            return $this->_outputDom;
        }

        return null;
    }

    /**
     * remove Commands and Supported from collections XML tree
     *
     * @param  DOMDocument $document
     * @return DOMDocument
     */
    protected function _cleanUpXML(DOMDocument $document)
    {
        $cleanedDocument = clone $document;

        $xpath = new DomXPath($cleanedDocument);
        $xpath->registerNamespace('AirSync', 'uri:AirSync');

        $collections = $xpath->query("//AirSync:Sync/AirSync:Collections/AirSync:Collection");

        // remove Commands and Supported elements
        foreach ($collections as $collection) {
            foreach (['Commands', 'Supported'] as $element) {
                /** @var DOMElement $collection */
                $childrenToRemove = $collection->getElementsByTagName($element);

                foreach ($childrenToRemove as $childToRemove) {
                    $collection->removeChild($childToRemove);
                }
            }
        }

        return $cleanedDocument;
    }

    /**
     * merge a partial XML document with the XML document from the previous request
     *
     * @param  DOMDocument|null  $requestBody
     * @return DOMDocument
     */
    protected function _mergeSyncRequest($requestBody, Syncroton_Model_Device $device)
    {
        $lastSyncCollection = [];

        if (!empty($device->lastsynccollection)) {
            $lastSyncCollection = Zend_Json::decode($device->lastsynccollection);
            if (!empty($lastSyncCollection['lastXML'])) {
                $lastXML = new DOMDocument();
                $lastXML->loadXML($lastSyncCollection['lastXML']);
            }
        }

        if (! $requestBody instanceof DOMDocument) {
            if (!empty($lastXML)) {
                $requestBody = $lastXML;
            } else {
                throw new Syncroton_Exception_UnexpectedValue('no xml body found');
            }
        }

        if ($requestBody->getElementsByTagName('Partial')->length > 0) {
            $partialBody = clone $requestBody;
            $requestBody = $lastXML ?? (new DOMDocument());

            $xpath = new DomXPath($requestBody);
            $xpath->registerNamespace('AirSync', 'uri:AirSync');

            foreach ($partialBody->documentElement->childNodes as $child) {
                if (! $child instanceof DOMElement) {
                    continue;
                }

                /** @var DOMElement $child */
                if ($child->tagName == 'Partial') {
                    continue;
                }

                if ($child->tagName == 'Collections') {
                    foreach ($child->getElementsByTagName('Collection') as $updatedCollection) {
                        $collectionId = $updatedCollection->getElementsByTagName('CollectionId')->item(0)->nodeValue;

                        $existingCollections = $xpath->query("//AirSync:Sync/AirSync:Collections/AirSync:Collection[AirSync:CollectionId='$collectionId']");

                        if ($existingCollections->length > 0) {
                            /** @var DOMElement $existingCollection */
                            $existingCollection = $existingCollections->item(0);
                            foreach ($updatedCollection->childNodes as $updatedCollectionChild) {
                                if (! $updatedCollectionChild instanceof DOMElement) {
                                    continue;
                                }

                                $duplicateChild = $existingCollection->getElementsByTagName($updatedCollectionChild->tagName);

                                if ($duplicateChild->length > 0) {
                                    $existingCollection->replaceChild($requestBody->importNode($updatedCollectionChild, true), $duplicateChild->item(0));
                                } else {
                                    $existingCollection->appendChild($requestBody->importNode($updatedCollectionChild, true));
                                }
                            }
                        } else {
                            $importedCollection = $requestBody->importNode($updatedCollection, true);
                        }
                    }

                } else {
                    $duplicateChild = $xpath->query("//AirSync:Sync/AirSync:{$child->tagName}");

                    if ($duplicateChild->length > 0) {
                        $requestBody->documentElement->replaceChild($requestBody->importNode($child, true), $duplicateChild->item(0));
                    } else {
                        $requestBody->documentElement->appendChild($requestBody->importNode($child, true));
                    }
                }
            }
        }

        $lastSyncCollection['lastXML'] = $this->_cleanUpXML($requestBody)->saveXML();

        $device->lastsynccollection = Zend_Json::encode($lastSyncCollection);

        return $requestBody;
    }
}
