<?php

/*
 +--------------------------------------------------------------------------+
 | Kolab Sync (ActiveSync for Kolab)                                        |
 |                                                                          |
 | Copyright (C) 2024, Apheleia IT AG <contact@apheleia-it.ch>              |
 |                                                                          |
 | This program is free software: you can redistribute it and/or modify     |
 | it under the terms of the GNU Affero General Public License as published |
 | by the Free Software Foundation, either version 3 of the License, or     |
 | (at your option) any later version.                                      |
 |                                                                          |
 | This program is distributed in the hope that it will be useful,          |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of           |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the             |
 | GNU Affero General Public License for more details.                      |
 |                                                                          |
 | You should have received a copy of the GNU Affero General Public License |
 | along with this program. If not, see <http://www.gnu.org/licenses/>      |
 +--------------------------------------------------------------------------+
 | Author: Christian Mollekopf <mollekopf@apheleia-it.ch>                      |
 +--------------------------------------------------------------------------+
*/

define('RCUBE_INSTALL_PATH', realpath(dirname(__FILE__) . '/../') . '/');
define('RCUBE_PLUGINS_DIR', RCUBE_INSTALL_PATH . 'lib/plugins/');

// Define include path
$include_path  = RCUBE_INSTALL_PATH . 'lib' . PATH_SEPARATOR;
$include_path .= RCUBE_INSTALL_PATH . 'lib/ext' . PATH_SEPARATOR;
$include_path .= ini_get('include_path');
set_include_path($include_path);

// include composer autoloader (if available)
if (@file_exists(RCUBE_INSTALL_PATH . 'vendor/autoload.php')) {
    require RCUBE_INSTALL_PATH . 'vendor/autoload.php';
}

// include global functions from Roundcube Framework
require_once 'Roundcube/bootstrap.php';

class SyncrotonCli
{
    public $rcube = null;
    public $userid = null;
    public $host;
    public $db;
    public $imap = null;
    public $sync = null;

    public function __construct()
    {
        $this->rcube = \rcube::get_instance();
        $host = $this->rcube->config->get('imap_host', $this->rcube->config->get('default_host'));
        $this->host = parse_url($host, PHP_URL_HOST) ?? $host;
        $this->port = parse_url($host, PHP_URL_PORT);
        $this->db = $this->rcube->get_dbh();
        ini_set('display_errors', 1);
        error_reporting(E_ALL);
        $this->sync = kolab_sync::get_instance();
    }


    public function connectToImap($email, $password, $adminpassword, $debug)
    {
        $proxyAuth = false;
        if ($adminpassword) {
            $proxyAuth = true;
            //FIXME will not work for dovecot
            $user = "cyrus-admin";
            $password = $adminpassword;
        }

        if (empty($password)) {
            rcube::raise_error("Password not specified (--adminpassword/--password).", false, true);
        }

        $imap = new \rcube_imap_generic();
        if ($proxyAuth) {
            $options['auth_cid'] = $user;
            $options['auth_pw'] = $password;
        }
        $options['auth_type'] = 'PLAIN';
        $options['port'] = $this->port;

        if ($this->port == 993) {
            $options['ssl_mode'] = 'ssl';
        }

        $options['socket_options'] = [
            'ssl' => [
                'verify_peer_name' => false,
                'verify_peer' => false,
                'allow_self_signed' => true,
            ],
        ];

        $imap->setDebug($debug);

        if (!$imap->connect($this->host, $email, $password, $options)) {
            rcube::raise_error("Failed to connect to imap.", false, true);
        }
        $this->imap = $imap;
        return $imap;
    }

    public function authenticate($password)
    {
        $this->rcube->set_user(new rcube_user($this->userid));
        if ($this->sync->authenticate($this->email, $password)) {
            $this->sync->password = $password;
        }
    }

    public function getStorage($password)
    {
        $this->authenticate($password);
        return $this->sync->storage();
    }

    public function getImapStorage($password)
    {
        $this->authenticate($password);
        return $this->sync->get_storage();
    }

    public function selectUser($user)
    {

        $select = $this->db->query(
            "SELECT `user_id` FROM `users`"
            . " WHERE `username` = ?"
            . " AND `mail_host` = ?"
            . " ORDER BY `user_id` DESC",
            \strtolower($user),
            \strtolower($this->host)
        );

        if ($data = $this->db->fetch_assoc($select)) {
            $this->userid = $data['user_id'];
        } else {
            \rcube::raise_error("User not found in Roundcube database: {$user}", false, true);
        }
        print("Found the user with id: {$this->userid}\n");

        $this->rcube->set_user(new rcube_user($this->userid));
        $this->email = $user;

        return $this->userid;
    }

    public function selectDevices($deviceid, $devicetype)
    {
        $devices = [];
        if (!empty($deviceid)) {
            $select = $this->db->query(
                "SELECT `id` FROM `syncroton_device` WHERE `owner_id` = ? AND ( `deviceid` = ? OR `id` = ? )",
                $this->userid,
                $deviceid,
                $deviceid
            );
        } elseif (!empty($devicetype)) {
            $select = $this->db->query(
                "SELECT `id` FROM `syncroton_device` WHERE `owner_id` = ? AND `devicetype` = ?",
                $this->userid,
                $devicetype
            );
        } else {
            $select = $this->db->query("SELECT `id` FROM `syncroton_device` WHERE `owner_id` = ?", $this->userid);
        }
        while ($record = $this->db->fetch_assoc($select)) {
            $devices[] = $record['id'];
        }
        return $devices;
    }

    // Translate from device.id to device.deviceid
    public function getDeviceId($id)
    {
        $select = $this->db->query(
            "SELECT `deviceid` FROM `syncroton_device` WHERE `id` = ?",
            $id
        );
        while ($record = $this->db->fetch_assoc($select)) {
            return $record['deviceid'];
        }
        return null;
    }

    public function setSubscriptions($id, $type, $subscriptions)
    {
        // This is essentially the same as kolab_subscriptions::set_subscriptions, but without the id translation.
        $data = json_encode($subscriptions);
        $db = $this->db;
        $query = $db->query("SELECT 1 FROM syncroton_subscriptions WHERE `device_id` = ? AND `type` = ?", $id, $type);

        if ($db->fetch_array($query)) {
            $query = $db->query("UPDATE syncroton_subscriptions SET `data` = ? WHERE `device_id` = ? AND `type` = ?", $data, $id, $type);
        } else {
            $query = $db->query("INSERT INTO syncroton_subscriptions (`device_id`, `type`, `data`) VALUES (?, ?, ?)", $id, $type, $data);
        }

        return $db->affected_rows($query) > 0;
    }

    public function getSubscriptions($id, $type)
    {
        $query = $this->db->query("SELECT `data` FROM syncroton_subscriptions WHERE `device_id` = ? AND `type` = ?", $id, $type);
        if ($record = $this->db->fetch_assoc($query)) {
            return json_decode($record['data'], true);
        }
        return [];
    }

};
