<?php

namespace Tests;

class SyncTestCase extends \PHPUnit\Framework\TestCase
{
    protected static ?\GuzzleHttp\Client $client;
    protected static ?string $deviceId;
    protected static ?string $deviceType;
    protected static ?string $host;
    protected static ?string $username;
    protected static ?string $password;
    protected static ?string $secondary_username;
    protected static ?string $secondary_password;
    protected static bool $authenticated = false;

    protected array $folders = [];

    /**
     * {@inheritDoc}
     */
    public function setUp(): void
    {
        if (empty(self::$username)) {
            $this->markTestSkipped('Not setup');
        }

        self::$deviceType = null;
    }

    /**
     * {@inheritDoc}
     */
    public static function setUpBeforeClass(): void
    {
        $sync = \kolab_sync::get_instance();
        $config = $sync->config;
        $db = $sync->get_dbh();

        self::$username = $config->get('activesync_test_username');
        self::$password = $config->get('activesync_test_password');
        self::$secondary_username = $config->get('activesync_test_secondary_username');
        self::$secondary_password = $config->get('activesync_test_secondary_password');
        self::$host = $config->get('activesync_test_host', 'http://localhost:8000');

        if (empty(self::$username)) {
            return;
        }

        self::$deviceId = 'test' . str_replace('.', '', microtime(true));

        $db->query('DELETE FROM syncroton_device');
        $db->query('DELETE FROM syncroton_data');
        $db->query('DELETE FROM syncroton_data_folder');

        self::$client = new \GuzzleHttp\Client([
                'http_errors' => false,
                'base_uri' => self::$host,
                'verify' => false,
                'auth' => [self::$username, self::$password],
                'connect_timeout' => 10,
                'timeout' => 10,
                'headers' => [
                    'Content-Type' => 'application/xml; charset=utf-8',
                    'Depth' => '1',
                ],
        ]);

        // TODO: execute: php -S localhost:8000
    }

    /**
     * {@inheritDoc}
     */
    public static function tearDownAfterClass(): void
    {
        if (self::$deviceId) {
            $sync = \kolab_sync::get_instance();
            /*
            if (self::$authenticated || $sync->authenticate(self::$username, self::$password)) {
                $sync->password = self::$password;
            }
            */

            $db = $sync->get_dbh();
            $db->query('DELETE FROM syncroton_device');
            $db->query('DELETE FROM syncroton_data');
            $db->query('DELETE FROM syncroton_data_folder');
        }
    }

    /**
     * Append an email message to the IMAP folder
     */
    protected function appendMail($folder, $filename, $replace = [])
    {
        $imap = $this->getImapStorage();

        $source = __DIR__ . '/src/' . $filename;

        if (!file_exists($source)) {
            exit("File does not exist: {$source}");
        }

        $is_file = true;

        if (!empty($replace)) {
            $is_file = false;
            $source = file_get_contents($source);
            foreach ($replace as $token => $value) {
                $source = str_replace($token, $value, $source);
            }
        }

        $uid = $imap->save_message($folder, $source, '', $is_file);

        if ($uid === false) {
            exit("Failed to append mail {$filename} into {$folder}");
        }

        return $uid;
    }

    /**
     * Run A SQL query
     */
    protected function runSQLQuery($query)
    {
        $sync = \kolab_sync::get_instance();
        $db = $sync->get_dbh();
        $db->query($query);
    }

    /**
     * Mark an email message as read over IMAP
     */
    protected function markMailAsRead($folder, $uids)
    {
        $imap = $this->getImapStorage();
        return $imap->set_flag($uids, 'SEEN', $folder);
    }

    /**
     * List emails over IMAP
     */
    protected function listEmails($folder, $uids)
    {
        $imap = $this->getImapStorage();
        return $imap->list_flags($folder, $uids);
    }

    /**
     * Append an DAV object to a DAV/IMAP folder
     */
    protected function appendObject($foldername, $filename, $type)
    {
        $path = __DIR__ . '/src/' . $filename;

        if (!file_exists($path)) {
            exit("File does not exist: {$path}");
        }

        $content = file_get_contents($path);
        $uid = preg_match('/UID:(?:urn:uuid:)?([a-z0-9-]+)/', $content, $m) ? $m[1] : null;

        if (empty($uid)) {
            exit("Filed to find UID in {$path}");
        }

        if ($this->isStorageDriver('kolab')) {
            $imap = $this->getImapStorage();
            if ($imap->folder_exists($foldername)) {
                // TODO
                exit("Not implemented for Kolab v3 storage driver");
            } else {
                exit("Folder is missing");
            }
        }

        $dav = $this->getDavStorage();

        foreach ($dav->get_folders($type) as $folder) {
            if ($folder->get_name() === $foldername) {
                $dav_type = $folder->get_dav_type();
                $location = $folder->object_location($uid);

                if ($folder->dav->create($location, $content, $dav_type) !== false) {
                    return;
                }
            }
        }

        exit("Failed to append object into {$foldername}");
    }

    /**
     * Delete a folder
     */
    protected function deleteTestFolder($name, $type)
    {
        // Deleting IMAP folders
        if ($type == 'mail' || $this->isStorageDriver('kolab')) {
            $imap = $this->getImapStorage();
            if ($imap->folder_exists($name)) {
                $imap->delete_folder($name);
            }

            return;
        }

        // Deleting DAV folders
        $dav = $this->getDavStorage();

        foreach ($dav->get_folders($type) as $folder) {
            if ($folder->get_name() === $name) {
                $dav->folder_delete($folder->id, $type);
            }
        }
    }

    /**
     * Create a folder
     */
    protected function createTestFolder($name, $type, $subscriptionState = '1', $subscribed = true)
    {
        // Create IMAP folders
        if ($type == 'mail' || $this->isStorageDriver('kolab')) {
            $imap = $this->getImapStorage();
            $imap->create_folder($name, $subscribed);
            if ($type != 'mail' && $this->isStorageDriver('kolab')) {
                $imap->set_metadata($name, ['/private/vendor/kolab/folder-type' => $type]);
            }
            $this->setSubscriptionState($name, $type, $subscriptionState);
        }
    }

    protected function createSharedTestFolder($otherUser, $name, $type, $subscriptionState = '1', $subscribed = true)
    {
        // Create IMAP folders
        if ($type == 'mail' || $this->isStorageDriver('kolab')) {
            $imap = $this->getImapStorage($otherUser);
            $imap->create_folder($name, $subscribed);
            if ($type != 'mail' && $this->isStorageDriver('kolab')) {
                $imap->set_metadata($name, ['/private/vendor/kolab/folder-type' => $type]);
            }
            $imap->set_acl($name, self::$username, 'lrswipkxt');
            $otherUserLocalPart = explode('@', $otherUser)[0];
            $this->setSubscriptionState("Other Users/$otherUserLocalPart/$name", $type, $subscriptionState);
        }
    }

    /**
     * Subscribe a test folder
     */
    public static function setSubscriptionState($name, $type, $subscriptionState, $deviceid = null)
    {
        $sync = \kolab_sync::get_instance();
        $db = $sync->get_dbh();
        $query = $db->query("SELECT `id` FROM `syncroton_device` WHERE `deviceid` = ?", $deviceid ?: self::$deviceId);
        $arr = $db->fetch_array($query);
        $device_id = $arr[0] ?? null;

        if (!$device_id) {
            throw new \Exception('Device ID not found');
        }

        [$type, ] = explode('.', $type);

        $query = $db->query(
            "SELECT `data` FROM `syncroton_subscriptions` WHERE `device_id` = ? AND `type` = ?",
            $device_id,
            $type
        );

        if ($record = $db->fetch_assoc($query)) {
            $data = json_decode($record['data'], true);
        } else {
            if (!$subscriptionState) {
                return;
            }
        }

        if ($subscriptionState) {
            $data[$name] = (int) $subscriptionState;
        } elseif (!isset($data[$name])) {
            return;
        } else {
            unset($data[$name]);
        }

        $data = json_encode($data);

        if ($record) {
            $db->query(
                'UPDATE syncroton_subscriptions SET `data` = ? WHERE `device_id` = ? AND `type` = ?',
                $data,
                $device_id,
                $type
            );
        } else {
            $db->query(
                'INSERT INTO `syncroton_subscriptions` (`data`, `device_id`, `type`) VALUES (?, ?, ?)',
                $data,
                $device_id,
                $type
            );
        }
    }

    public static function wipeSubscriptionState($type, $deviceid = null)
    {
        $sync = \kolab_sync::get_instance();
        $db = $sync->get_dbh();
        $query = $db->query("SELECT `id` FROM `syncroton_device` WHERE `deviceid` = ?", $deviceid ?: self::$deviceId);
        $arr = $db->fetch_array($query);
        $device_id = $arr[0] ?? null;

        $db->query(
            'DELETE FROM `syncroton_subscriptions` WHERE `device_id` = ? AND `type` = ?',
            $device_id,
            $type
        );
    }

    /**
     * Remove all objects from a folder
     */
    protected function emptyTestFolder($name, $type)
    {
        // Deleting in IMAP folders
        if ($type == 'mail' || $this->isStorageDriver('kolab')) {
            $imap = $this->getImapStorage();
            $imap->delete_message('*', $name);
            return;
        }

        // Deleting in DAV folders
        $dav = $this->getDavStorage();

        foreach ($dav->get_folders($type) as $folder) {
            if ($folder->get_name() === $name) {
                $folder->delete_all();
            }
        }
    }

    /**
     * Convert WBXML binary content into XML
     */
    protected function fromWbxml($binary)
    {
        $stream = fopen('php://memory', 'r+');
        fwrite($stream, $binary);
        rewind($stream);
        $decoder = new \Syncroton_Wbxml_Decoder($stream);

        return $decoder->decode();
    }

    /**
     * Get objects from a DAV folder
     */
    protected function getDavObjects($foldername, $type, $query = [])
    {
        $dav = $this->getDavStorage();

        foreach ($dav->get_folders($type) as $folder) {
            if ($folder->get_name() === $foldername) {
                $result = [];
                foreach ($folder->select($query) as $object) {
                    $result[] = $object;
                }
                return $result;
            }
        }

        throw new \Exception("Folder not found");
    }

    /**
     * Initialize DAV storage
     */
    protected function getDavStorage()
    {
        $sync = \kolab_sync::get_instance();
        $url = $sync->config->get('activesync_dav_server', 'http://localhost');

        if (strpos($url, '://') === false) {
            $url = 'http://' . $url;
        }

        // Inject user+password to the URL, there's no other way to pass it to the DAV client
        $url = str_replace('://', '://' . rawurlencode(self::$username) . ':' . rawurlencode(self::$password) . '@', $url);

        // Make sure user is authenticated
        $this->getImapStorage();
        if ($sync->user) {
            // required e.g. for DAV client cache use
            \rcube::get_instance()->user = $sync->user;
        }

        return new \kolab_storage_dav($url);
    }

    /**
     * Initialize IMAP storage
     */
    protected function getImapStorage($user = null)
    {
        $sync = \kolab_sync::get_instance();

        if ($user || !self::$authenticated) {
            if (!$user) {
                $user = self::$username;
            }
            if ($sync->authenticate($user, self::$password)) {
                self::$authenticated = true;
                $sync->password = self::$password;
            }
        }

        return $sync->get_storage();
    }

    /**
     * Check the configured activesync_storage driver
     */
    protected function isStorageDriver($name)
    {
        return $name === \kolab_sync::get_instance()->config->get('activesync_storage', 'kolab');
    }

    /**
     * Make a HTTP request to the ActiveSync server
     */
    protected function request($body, $cmd, $type = 'POST')
    {
        $username = self::$username;
        $deviceId = self::$deviceId;
        $deviceType = self::$deviceType ?: 'WindowsOutlook15';

        $body = $this->toWbxml($body);

        return self::$client->request(
            $type,
            "?Cmd={$cmd}&User={$username}&DeviceId={$deviceId}&DeviceType={$deviceType}",
            [
                'headers' => [
                    'Content-Type' => 'application/vnd.ms-sync.wbxml',
                    'MS-ASProtocolVersion' => '14.0',
                ],
                'body' => $body,
            ]
        );
    }

    /**
     * Register the device for tests, some commands do not work until device/folders are registered
     */
    protected function registerDevice()
    {
        // Execute initial FolderSync, it is required before executing some commands
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        foreach ($xpath->query("//ns:FolderSync/ns:Changes/ns:Add") as $idx => $folder) {
            $serverId = $folder->getElementsByTagName('ServerId')->item(0)->nodeValue;
            $displayName = $folder->getElementsByTagName('DisplayName')->item(0)->nodeValue;
            $this->folders[$serverId] = $displayName;
        }
    }

    protected function resetDevice()
    {
        $sync = \kolab_sync::get_instance();

        $db = $sync->get_dbh();
        $db->query('DELETE FROM syncroton_device');
    }

    /**
     * Convert XML into WBXML binary content
     */
    protected function toWbxml($xml)
    {
        $outputStream = fopen('php://temp', 'r+');
        $encoder = new \Syncroton_Wbxml_Encoder($outputStream, 'UTF-8', 3);
        $dom = new \DOMDocument();
        $dom->loadXML($xml);
        $encoder->encode($dom);
        rewind($outputStream);

        return stream_get_contents($outputStream);
    }

    /**
     * Get XPath from a DOM
     */
    protected function xpath($dom)
    {
        $xpath = new \DOMXpath($dom);
        $xpath->registerNamespace("ns", $dom->documentElement->namespaceURI);
        $xpath->registerNamespace("AirSync", "uri:AirSync");
        $xpath->registerNamespace("AirSyncBase", "uri:AirSyncBase");
        $xpath->registerNamespace("Calendar", "uri:Calendar");
        $xpath->registerNamespace("Contacts", "uri:Contacts");
        $xpath->registerNamespace("ComposeMail", "uri:ComposeMail");
        $xpath->registerNamespace("Email", "uri:Email");
        $xpath->registerNamespace("Email2", "uri:Email2");
        $xpath->registerNamespace("Settings", "uri:Settings");
        $xpath->registerNamespace("Tasks", "uri:Tasks");

        return $xpath;
    }

    /**
      * Pretty print the DOM
      */
    protected function printDom($dom)
    {
        $dom->formatOutput = true;
        print($dom->saveXML());
    }
}
