<?php

class MoveItemsTest extends Tests\SyncTestCase
{
    /**
     * Test moving an email message
     */
    public function testMoveEmail()
    {
        $this->emptyTestFolder('INBOX', 'mail');
        $this->emptyTestFolder('Trash', 'mail');
        $uid = $this->appendMail('INBOX', 'mail.sync1');
        $this->registerDevice();

        $inbox = array_search('INBOX', $this->folders);
        $trash = array_search('Trash', $this->folders);

        // Initial sync
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync">
                <Collections>
                    <Collection>
                        <SyncKey>0</SyncKey>
                        <CollectionId>{$inbox}</CollectionId>
                    </Collection>
                    <Collection>
                        <SyncKey>0</SyncKey>
                        <CollectionId>{$trash}</CollectionId>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        // Sync mail from INBOX and Trash
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase">
                <Collections>
                    <Collection>
                        <SyncKey>1</SyncKey>
                        <CollectionId>{$inbox}</CollectionId>
                        <GetChanges>1</GetChanges>
                    </Collection>
                    <Collection>
                        <SyncKey>1</SyncKey>
                        <CollectionId>{$trash}</CollectionId>
                        <GetChanges>1</GetChanges>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame(1, $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->count());
        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(0);
        $this->assertSame(1, $xpath->query("ns:Commands/ns:Add", $root)->count());
        $root = $xpath->query("ns:Commands/ns:Add", $root)->item(0);
        $this->assertSame('test sync', $xpath->query("ns:ApplicationData/Email:Subject", $root)->item(0)->nodeValue);

        // Move the message to $trash
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <MoveItems xmlns="uri:Move">
                <Move>
                    <SrcMsgId>{$inbox}::{$uid}</SrcMsgId>
                    <SrcFldId>{$inbox}</SrcFldId>
                    <DstFldId>{$trash}</DstFldId>
                </Move>
            </MoveItems>
            EOF;

        $response = $this->request($request, 'MoveItems');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $xpath->registerNamespace('Move', 'uri:Move');

        $root = $xpath->query("//Move:MoveItems/Move:Response")->item(0);
        $this->assertSame('3', $xpath->query("Move:Status", $root)->item(0)->nodeValue);
        $this->assertSame("{$inbox}::{$uid}", $xpath->query("Move:SrcMsgId", $root)->item(0)->nodeValue);
        $serverId = $xpath->query("Move:DstMsgId", $root)->item(0)->nodeValue;

        // Sync mail from INBOX and Trash
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase">
                <Collections>
                    <Collection>
                        <SyncKey>2</SyncKey>
                        <CollectionId>{$inbox}</CollectionId>
                        <GetChanges>1</GetChanges>
                    </Collection>
                    <Collection>
                        <SyncKey>1</SyncKey>
                        <CollectionId>{$trash}</CollectionId>
                        <GetChanges>1</GetChanges>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(0); // INBOX
        $this->assertSame($inbox, $xpath->query("ns:CollectionId", $root)->item(0)->nodeValue);
        $this->assertSame(1, $xpath->query("ns:Commands/ns:Delete", $root)->count());
        $this->assertSame("$inbox::$uid", $xpath->query("ns:Commands/ns:Delete/ns:ServerId", $root)->item(0)->nodeValue);

        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(1); // Trash
        $this->assertSame($trash, $xpath->query("ns:CollectionId", $root)->item(0)->nodeValue);
        $this->assertSame(1, $xpath->query("ns:Commands/ns:Add", $root)->count());
        $this->assertSame('test sync', $xpath->query("ns:Commands/ns:Add/ns:ApplicationData/Email:Subject", $root)->item(0)->nodeValue);
    }

    public function testInvalidMove()
    {
        $this->emptyTestFolder('INBOX', 'mail');
        $this->emptyTestFolder('Trash', 'mail');
        $uid = $this->appendMail('INBOX', 'mail.sync1');
        $this->registerDevice();
        $inbox = array_search('INBOX', $this->folders);
        $trash = array_search('Trash', $this->folders);

        // Move item that doesn't exist
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <MoveItems xmlns="uri:Move">
                <Move>
                    <SrcMsgId>foobar::99999</SrcMsgId>
                    <SrcFldId>foobar</SrcFldId>
                    <DstFldId>foobar</DstFldId>
                </Move>
            </MoveItems>
            EOF;

        $response = $this->request($request, 'MoveItems');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $xpath->registerNamespace('Move', 'uri:Move');

        $root = $xpath->query("//Move:MoveItems/Move:Response")->item(0);
        $this->assertSame('1', $xpath->query("Move:Status", $root)->item(0)->nodeValue);

        // Move item that doesn't exist
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <MoveItems xmlns="uri:Move">
                <Move>
                    <SrcMsgId>{$inbox}::99999</SrcMsgId>
                    <SrcFldId>{$inbox}</SrcFldId>
                    <DstFldId>{$trash}</DstFldId>
                </Move>
            </MoveItems>
            EOF;

        $response = $this->request($request, 'MoveItems');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $xpath->registerNamespace('Move', 'uri:Move');

        $root = $xpath->query("//Move:MoveItems/Move:Response")->item(0);
        $this->assertSame('1', $xpath->query("Move:Status", $root)->item(0)->nodeValue);
    }

    /**
     * Test moving a contact
     */
    public function testMoveContact()
    {
        if ($this->isStorageDriver('kolab')) {
            // The Contacts folder is not available, and consequently appendObject fails
            $this->markTestSkipped('This test only works with the DAV backend.');
        }

        // Test with multi-folder support enabled
        self::$deviceType = 'iphone';

        $davFolder = 'Contacts';
        $this->emptyTestFolder($davFolder, 'contact');
        $this->deleteTestFolder($folderName = 'Test Contacts Folder', 'contact');
        $this->appendObject($davFolder, 'contact.vcard1', 'contact');

        $this->registerDevice();

        $srcFolderId = array_search($davFolder, $this->folders);
        $this->assertTrue(!empty($srcFolderId));

        // Create a contacts folder
        $folderType = Syncroton_Command_FolderSync::FOLDERTYPE_CONTACT_USER_CREATED;
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderCreate xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
                <ParentId>0</ParentId>
                <DisplayName>{$folderName}</DisplayName>
                <Type>{$folderType}</Type>
            </FolderCreate>
            EOF;

        $response = $this->request($request, 'FolderCreate');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $dstFolderId = $xpath->query("//ns:FolderCreate/ns:ServerId")->item(0)->nodeValue;

        // Sync both folders
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase">
                <Collections>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>0</SyncKey>
                        <CollectionId>{$srcFolderId}</CollectionId>
                    </Collection>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>0</SyncKey>
                        <CollectionId>{$dstFolderId}</CollectionId>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());
        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(0);
        $this->assertSame('1', $xpath->query("ns:Status", $root)->item(0)->nodeValue);

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase">
                <Collections>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>1</SyncKey>
                        <CollectionId>{$srcFolderId}</CollectionId>
                        <DeletesAsMoves/>
                        <GetChanges/>
                        <Options>
                            <AirSyncBase:BodyPreference>
                                <AirSyncBase:Type>1</AirSyncBase:Type>
                                <AirSyncBase:TruncationSize>5120</AirSyncBase:TruncationSize>
                            </AirSyncBase:BodyPreference>
                            <Conflict>1</Conflict>
                        </Options>
                    </Collection>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>1</SyncKey>
                        <CollectionId>{$dstFolderId}</CollectionId>
                        <DeletesAsMoves/>
                        <GetChanges/>
                        <Options>
                            <AirSyncBase:BodyPreference>
                                <AirSyncBase:Type>1</AirSyncBase:Type>
                                <AirSyncBase:TruncationSize>5120</AirSyncBase:TruncationSize>
                            </AirSyncBase:BodyPreference>
                            <Conflict>1</Conflict>
                        </Options>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(0);
        $this->assertSame('1', $xpath->query("ns:Status", $root)->item(0)->nodeValue);
        $this->assertSame($srcFolderId, $xpath->query("ns:CollectionId", $root)->item(0)->nodeValue);
        $this->assertSame(1, $xpath->query("ns:Commands/ns:Add", $root)->count());
        $this->assertSame('Jane', $xpath->query("ns:Commands/ns:Add/ns:ApplicationData/Contacts:FirstName", $root)->item(0)->nodeValue);
        $srcMsgId = $xpath->query("ns:Commands/ns:Add/ns:ServerId", $root)->item(0)->nodeValue;

        // Move the message to the other folder
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <MoveItems xmlns="uri:Move">
                <Move>
                    <SrcMsgId>{$srcMsgId}</SrcMsgId>
                    <SrcFldId>{$srcFolderId}</SrcFldId>
                    <DstFldId>{$dstFolderId}</DstFldId>
                </Move>
            </MoveItems>
            EOF;

        $response = $this->request($request, 'MoveItems');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $xpath->registerNamespace('Move', 'uri:Move');

        $root = $xpath->query("//Move:MoveItems/Move:Response")->item(0);
        $this->assertSame('3', $xpath->query("Move:Status", $root)->item(0)->nodeValue);
        $this->assertSame($srcMsgId, $xpath->query("Move:SrcMsgId", $root)->item(0)->nodeValue);
        $dstMsgId = $xpath->query("Move:DstMsgId", $root)->item(0)->nodeValue;

        // Sync the folders again
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase">
                <Collections>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>2</SyncKey>
                        <CollectionId>{$srcFolderId}</CollectionId>
                        <DeletesAsMoves/>
                        <GetChanges/>
                        <Options>
                            <AirSyncBase:BodyPreference>
                                <AirSyncBase:Type>1</AirSyncBase:Type>
                                <AirSyncBase:TruncationSize>5120</AirSyncBase:TruncationSize>
                            </AirSyncBase:BodyPreference>
                            <Conflict>1</Conflict>
                        </Options>
                    </Collection>
                    <Collection>
                        <Class>Contacts</Class>
                        <SyncKey>1</SyncKey>
                        <CollectionId>{$dstFolderId}</CollectionId>
                        <DeletesAsMoves/>
                        <GetChanges/>
                        <Options>
                            <AirSyncBase:BodyPreference>
                                <AirSyncBase:Type>1</AirSyncBase:Type>
                                <AirSyncBase:TruncationSize>5120</AirSyncBase:TruncationSize>
                            </AirSyncBase:BodyPreference>
                            <Conflict>1</Conflict>
                        </Options>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(0); // src folder
        $this->assertSame($srcFolderId, $xpath->query("ns:CollectionId", $root)->item(0)->nodeValue);
        $this->assertSame(1, $xpath->query("ns:Commands/ns:Delete", $root)->count());
        $this->assertSame($srcMsgId, $xpath->query("ns:Commands/ns:Delete/ns:ServerId", $root)->item(0)->nodeValue);

        $root = $xpath->query("//ns:Sync/ns:Collections/ns:Collection")->item(1); // dst folder
        $this->assertSame($dstFolderId, $xpath->query("ns:CollectionId", $root)->item(0)->nodeValue);
        $this->assertSame(1, $xpath->query("ns:Commands/ns:Add", $root)->count());
        $this->assertSame('Jane', $xpath->query("ns:Commands/ns:Add/ns:ApplicationData/Contacts:FirstName", $root)->item(0)->nodeValue);
        $this->assertSame($dstMsgId, $xpath->query("ns:Commands/ns:Add/ns:ServerId", $root)->item(0)->nodeValue);

        $this->deleteTestFolder($folderName, 'contact');
    }
}
