<?php

/**
 * Zend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @package    Syncroton
 * @copyright  Copyright (c) 2005-2009 Zend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @version    $Id: Registry.php 10020 2009-08-18 14:34:09Z j.fischer@metaways.de $
 */

/**
 * Generic storage class helps to manage global data.
 *
 * @package    Syncroton
 * @copyright  Copyright (c) 2005-2009 Zend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 */
class Syncroton_Registry extends ArrayObject
{
    public const CALENDAR_DATA_CLASS = 'calendar_data_class';
    public const CONTACTS_DATA_CLASS = 'contacts_data_class';
    public const EMAIL_DATA_CLASS    = 'email_data_class';
    public const NOTES_DATA_CLASS    = 'notes_data_class';
    public const TASKS_DATA_CLASS    = 'tasks_data_class';
    public const GAL_DATA_CLASS      = 'gal_data_class';

    public const DEFAULT_POLICY      = 'default_policy';
    public const PING_TIMEOUT        = 'ping_timeout';
    public const PING_INTERVAL       = 'ping_interval';
    public const QUIET_TIME          = 'quiet_time';
    public const SESSION_VALIDATOR   = 'session_validator';
    public const MAX_COLLECTIONS     = 'max_collections';
    public const MAX_PING_INTERVAL   = 'max_ping_interval';
    public const BLOCK_BROKEN_DEVICES = 'block_broken_devices';

    public const DATABASE            = 'database';
    public const TRANSACTIONMANAGER  = 'transactionmanager';

    public const CONTENTSTATEBACKEND = 'contentstatebackend';
    public const DEVICEBACKEND       = 'devicebackend';
    public const FOLDERBACKEND       = 'folderbackend';
    public const POLICYBACKEND       = 'policybackend';
    public const SYNCSTATEBACKEND    = 'syncstatebackend';
    public const LOGGERBACKEND       = 'loggerBackend';

    public const SLEEP_CALLBACK      = 'sleep_callback';
    public const WAKEUP_CALLBACK     = 'wakeup_callback';

    /**
     * Class name of the singleton registry object.
     * @var string
     */
    private static $_registryClassName = 'Syncroton_Registry';

    /**
     * Registry object provides storage for shared objects.
     * @var Syncroton_Registry|null
     */
    private static $_registry = null;

    /**
     * Retrieves the default registry instance.
     *
     * @return Syncroton_Registry
     */
    public static function getInstance()
    {
        if (self::$_registry === null) {
            self::init();
        }

        return self::$_registry;
    }

    /**
     * @return Zend_Db_Adapter_Abstract
     */
    public static function getDatabase()
    {
        return self::get(self::DATABASE);
    }

    /**
     * return transaction manager class
     *
     * @return Syncroton_TransactionManagerInterface
     */
    public static function getTransactionManager()
    {
        if (!self::isRegistered(self::TRANSACTIONMANAGER)) {
            self::set(self::TRANSACTIONMANAGER, Syncroton_TransactionManager::getInstance());
        }

        return self::get(self::TRANSACTIONMANAGER);
    }

    /**
     * Set the default registry instance to a specified instance.
     *
     * @param Syncroton_Registry $registry An object instance of type Syncroton_Registry,
     *   or a subclass.
     * @return void
     * @throws Zend_Exception if registry is already initialized.
     */
    public static function setInstance(Syncroton_Registry $registry)
    {
        if (self::$_registry !== null) {
            require_once 'Zend/Exception.php';
            throw new Zend_Exception('Registry is already initialized');
        }

        self::setClassName(get_class($registry));
        self::$_registry = $registry;
    }

    /**
     * Initialize the default registry instance.
     *
     * @return void
     */
    protected static function init()
    {
        self::setInstance(new self::$_registryClassName());
    }

    /**
     * Set the class name to use for the default registry instance.
     * Does not affect the currently initialized instance, it only applies
     * for the next time you instantiate.
     *
     * @param string $registryClassName
     * @return void
     * @throws Zend_Exception if the registry is initialized or if the
     *   class name is not valid.
     */
    public static function setClassName($registryClassName = 'Syncroton_Registry')
    {
        if (self::$_registry !== null) {
            require_once 'Zend/Exception.php';
            throw new Zend_Exception('Registry is already initialized');
        }

        if (!is_string($registryClassName)) {
            require_once 'Zend/Exception.php';
            throw new Zend_Exception("Argument is not a class name");
        }

        /**
         * @see Zend_Loader
         */
        if (!class_exists($registryClassName)) {
            require_once 'Zend/Loader.php';
            Zend_Loader::loadClass($registryClassName);
        }

        self::$_registryClassName = $registryClassName;
    }

    /**
     * Unset the default registry instance.
     * Primarily used in tearDown() in unit tests.
     * @returns void
     */
    public static function _unsetInstance()
    {
        self::$_registry = null;
    }

    /**
     * getter method, basically same as offsetGet().
     *
     * This method can be called from an object of type Syncroton_Registry, or it
     * can be called statically.  In the latter case, it uses the default
     * static instance stored in the class.
     *
     * @param string $index - get the value associated with $index
     * @return mixed
     * @throws Zend_Exception if no entry is registerd for $index.
     */
    public static function get($index)
    {
        $instance = self::getInstance();

        if (!$instance->offsetExists($index)) {
            require_once 'Zend/Exception.php';
            throw new Zend_Exception("No entry is registered for key '$index'");
        }

        return $instance->offsetGet($index);
    }

    /**
     * returns content state backend
     *
     * creates Syncroton_Backend_Content on the fly if not before via
     * Syncroton_Registry::set(self::CONTENTSTATEBACKEND, $backend);
     *
     * @return Syncroton_Backend_IContent
     */
    public static function getContentStateBackend()
    {
        if (!self::isRegistered(self::CONTENTSTATEBACKEND)) {
            self::set(self::CONTENTSTATEBACKEND, new Syncroton_Backend_Content(self::getDatabase()));
        }

        return self::get(self::CONTENTSTATEBACKEND);
    }

    /**
     * returns device backend
     *
     * creates Syncroton_Backend_Device on the fly if not before via
     * Syncroton_Registry::set(self::DEVICEBACKEND, $backend);
     *
     * @return Syncroton_Backend_IDevice
     */
    public static function getDeviceBackend()
    {
        if (!self::isRegistered(self::DEVICEBACKEND)) {
            self::set(self::DEVICEBACKEND, new Syncroton_Backend_Device(self::getDatabase()));
        }

        return self::get(self::DEVICEBACKEND);
    }

    /**
     * returns folder backend
     *
     * creates Syncroton_Backend_Folder on the fly if not before via
     * Syncroton_Registry::set(self::FOLDERBACKEND, $backend);
     *
     * @return Syncroton_Backend_IFolder
     */
    public static function getFolderBackend()
    {
        if (!self::isRegistered(self::FOLDERBACKEND)) {
            self::set(self::FOLDERBACKEND, new Syncroton_Backend_Folder(self::getDatabase()));
        }

        return self::get(self::FOLDERBACKEND);
    }

    /**
     * Return maximum ping interval (HeartbeatInterval) value (in seconds)
     *
     * @return int
     */
    public static function getPingInterval()
    {
        if (!self::isRegistered(self::PING_INTERVAL)) {
            return 3540; // 59 minutes limit defined in Activesync protocol spec.
        }

        return self::get(self::PING_INTERVAL);
    }

    /**
    /**
     * Return maximum ping interval (HeartbeatInterval) value (in seconds)
     *
     * @return int
     */
    public static function getMaxPingInterval()
    {
        if (!self::isRegistered(self::MAX_PING_INTERVAL)) {
            return Syncroton_Command_Ping::MAX_PING_INTERVAL;
        }

        return self::get(self::MAX_PING_INTERVAL);
    }

    /**
     * return ping timeout
     *
     * sleep "ping timeout" seconds between folder checks in Ping and Sync command
     *
     * @return int
     */
    public static function getPingTimeout()
    {
        if (!self::isRegistered(self::PING_TIMEOUT)) {
            return 60;
        }

        return self::get(self::PING_TIMEOUT);
    }

    /**
     * Return maximum number of collections in Sync/Ping request
     *
     * @return int
     */
    public static function getMaxCollections()
    {
        return self::get(self::MAX_COLLECTIONS);
    }

    /**
     * returns policy backend
     *
     * creates Syncroton_Backend_Policy on the fly if not set before via
     * Syncroton_Registry::set(self::POLICYBACKEND, $backend);
     *
     * @return Syncroton_Backend_Policy
     */
    public static function getPolicyBackend()
    {
        if (!self::isRegistered(self::POLICYBACKEND)) {
            self::set(self::POLICYBACKEND, new Syncroton_Backend_Policy(self::getDatabase()));
        }

        return self::get(self::POLICYBACKEND);
    }

    /**
     * return quiet time
     *
     * don't check folders if last sync was "quiet time" seconds ago
     *
     * @return int
     */
    public static function getQuietTime()
    {
        if (!self::isRegistered(self::QUIET_TIME)) {
            return 180;
        }

        return self::get(self::QUIET_TIME);
    }

    /**
     * Returns sleep callback function
     *
     * This function is used in long running requests like ping & sync to
     * close connections to external sources (e.g. database) before we
     * call sleep() to wait some time for next iteration.
     * Callback should throw exceptions on errors.
     *
     * @return callable
     */
    public static function getSleepCallback()
    {
        if (!self::isRegistered(self::SLEEP_CALLBACK)) {
            self::set(self::SLEEP_CALLBACK, function () {});
        }

        return self::get(self::SLEEP_CALLBACK);
    }

    /**
     * Returns wakeup callback function
     *
     * This function is used in long running requests like ping & sync to
     * re-connect to external sources (e.g. database) closed by sleep callback.
     * Callback should throw exceptions on errors.
     *
     * @return callable
     */
    public static function getWakeupCallback()
    {
        if (!self::isRegistered(self::WAKEUP_CALLBACK)) {
            self::set(self::WAKEUP_CALLBACK, function () {});
        }

        return self::get(self::WAKEUP_CALLBACK);
    }

    /**
     * return session validation function
     *
     * This function is used in long running requests like ping & sync to
     * validate user session. Returns false if session is not valid any longer
     *
     * @return callable
     */
    public static function getSessionValidator()
    {
        if (!self::isRegistered(self::SESSION_VALIDATOR)) {
            self::set(self::SESSION_VALIDATOR, function () {
                return true;
            });
        }

        return self::get(self::SESSION_VALIDATOR);
    }

    /**
     * returns syncstate backend
     *
     * creates Syncroton_Backend_SyncState on the fly if not before via
     * Syncroton_Registry::set(self::SYNCSTATEBACKEND, $backend);
     *
     * @return Syncroton_Backend_ISyncState
     */
    public static function getSyncStateBackend()
    {
        if (!self::isRegistered(self::SYNCSTATEBACKEND)) {
            self::set(self::SYNCSTATEBACKEND, new Syncroton_Backend_SyncState(self::getDatabase()));
        }

        return self::get(self::SYNCSTATEBACKEND);
    }

    /**
     * setter method, basically same as offsetSet().
     *
     * This method can be called from an object of type Syncroton_Registry, or it
     * can be called statically.  In the latter case, it uses the default
     * static instance stored in the class.
     *
     * @param string $index The location in the ArrayObject in which to store
     *   the value.
     * @param mixed $value The object to store in the ArrayObject.
     * @return void
     */
    public static function set($index, $value)
    {
        $instance = self::getInstance();
        $instance->offsetSet($index, $value);
    }

    public static function setDatabase(Zend_Db_Adapter_Abstract $db)
    {
        self::set(self::DATABASE, $db);
    }

    public static function setCalendarDataClass($className)
    {
        if (!class_exists($className)) {
            throw new InvalidArgumentException('invalid $_className provided');
        }

        self::set(self::CALENDAR_DATA_CLASS, $className);
    }

    public static function setContactsDataClass($className)
    {
        if (!class_exists($className)) {
            throw new InvalidArgumentException('invalid $_className provided');
        }

        self::set(self::CONTACTS_DATA_CLASS, $className);
    }

    public static function setEmailDataClass($className)
    {
        if (!class_exists($className)) {
            throw new InvalidArgumentException('invalid $_className provided');
        }

        self::set(self::EMAIL_DATA_CLASS, $className);
    }

    public static function setNotesDataClass($className)
    {
        if (!class_exists($className)) {
            throw new InvalidArgumentException('invalid $_className provided');
        }

        self::set(self::NOTES_DATA_CLASS, $className);
    }

    public static function setTasksDataClass($className)
    {
        if (!class_exists($className)) {
            throw new InvalidArgumentException('invalid $_className provided');
        }

        self::set(self::TASKS_DATA_CLASS, $className);
    }

    public static function setGALDataClass($className)
    {
        if (!class_exists($className)) {
            throw new InvalidArgumentException('invalid $_className provided');
        }

        self::set(self::GAL_DATA_CLASS, $className);
    }

    public static function setTransactionManager($manager)
    {
        self::set(self::TRANSACTIONMANAGER, $manager);
    }

    /**
     * Returns TRUE if the $index is a named value in the registry,
     * or FALSE if $index was not found in the registry.
     *
     * @param  string $index
     * @return boolean
     */
    public static function isRegistered($index)
    {
        if (self::$_registry === null) {
            return false;
        }
        return self::$_registry->offsetExists($index);
    }

    /**
     * Constructs a parent ArrayObject with default
     * ARRAY_AS_PROPS to allow acces as an object
     *
     * @param array $array data array
     * @param integer $flags ArrayObject flags
     */
    public function __construct($array = [], $flags = parent::ARRAY_AS_PROPS)
    {
        parent::__construct($array, $flags);
    }
}
