<?php

/**
 * Syncroton
 *
 * @package     Syncroton
 * @subpackage  Command
 * @license     http://www.tine20.org/licenses/lgpl.html LGPL Version 3
 * @copyright   Copyright (c) 2008-2012 Metaways Infosystems GmbH (http://www.metaways.de)
 * @author      Lars Kneschke <l.kneschke@metaways.de>
 */

/**
 * class to handle ActiveSync FolderSync command
 *
 * @package     Syncroton
 * @subpackage  Command
 */
class Syncroton_Command_FolderSync extends Syncroton_Command_Wbxml
{
    public const STATUS_SUCCESS                = 1;
    public const STATUS_FOLDER_EXISTS          = 2;
    public const STATUS_IS_SPECIAL_FOLDER      = 3;
    public const STATUS_FOLDER_NOT_FOUND       = 4;
    public const STATUS_PARENT_FOLDER_NOT_FOUND = 5;
    public const STATUS_SERVER_ERROR           = 6;
    public const STATUS_ACCESS_DENIED          = 7;
    public const STATUS_REQUEST_TIMED_OUT      = 8;
    public const STATUS_INVALID_SYNC_KEY       = 9;
    public const STATUS_MISFORMATTED           = 10;
    public const STATUS_UNKNOWN_ERROR          = 11;

    /**
     * some usefull constants for working with the xml files
     */
    public const FOLDERTYPE_GENERIC_USER_CREATED   = 1;
    public const FOLDERTYPE_INBOX                  = 2;
    public const FOLDERTYPE_DRAFTS                 = 3;
    public const FOLDERTYPE_DELETEDITEMS           = 4;
    public const FOLDERTYPE_SENTMAIL               = 5;
    public const FOLDERTYPE_OUTBOX                 = 6;
    public const FOLDERTYPE_TASK                   = 7;
    public const FOLDERTYPE_CALENDAR               = 8;
    public const FOLDERTYPE_CONTACT                = 9;
    public const FOLDERTYPE_NOTE                   = 10;
    public const FOLDERTYPE_JOURNAL                = 11;
    public const FOLDERTYPE_MAIL_USER_CREATED      = 12;
    public const FOLDERTYPE_CALENDAR_USER_CREATED  = 13;
    public const FOLDERTYPE_CONTACT_USER_CREATED   = 14;
    public const FOLDERTYPE_TASK_USER_CREATED      = 15;
    public const FOLDERTYPE_JOURNAL_USER_CREATED   = 16;
    public const FOLDERTYPE_NOTE_USER_CREATED      = 17;
    public const FOLDERTYPE_UNKOWN                 = 18;

    protected $_defaultNameSpace    = 'uri:FolderHierarchy';
    protected $_documentElement     = 'FolderSync';

    protected $_classes             = [
        Syncroton_Data_Factory::CLASS_CALENDAR,
        Syncroton_Data_Factory::CLASS_CONTACTS,
        Syncroton_Data_Factory::CLASS_EMAIL,
        Syncroton_Data_Factory::CLASS_NOTES,
        Syncroton_Data_Factory::CLASS_TASKS,
    ];

    /**
     * @var string
     */
    protected $_syncKey;

    /**
     * @var bool
     */
    protected $_syncKeyReused = false;

    /**
     * Parse FolderSync request
     */
    public function handle()
    {
        $xml = simplexml_import_dom($this->_requestBody);
        $syncKey = (int)$xml->SyncKey;

        if ($this->_logger instanceof Zend_Log) {
            $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " synckey is $syncKey");
        }

        if ($syncKey === 0) {
            $this->_syncState = new Syncroton_Model_SyncState([
                'device_id' => $this->_device,
                'counter'   => 0,
                'type'      => 'FolderSync',
                'lastsync'  => $this->_syncTimeStamp,
            ]);

            // reset state of foldersync
            $this->_syncStateBackend->resetState($this->_device, 'FolderSync');

            return;
        } else {
            // The synckey that is sent to us should already be existing, because we create it at the end,
            // however, the next one shouldn't. When it does exist it means that the same request has been resent,
            // which can happen if the client failed to receive the response.
            if ($this->_syncStateBackend->haveNext($this->_device, 'FolderSync', $syncKey)) {
                $this->_syncKeyReused = true;
            }
        }
        if (!($this->_syncState = $this->_syncStateBackend->validate($this->_device, 'FolderSync', $syncKey)) instanceof Syncroton_Model_SyncState) {
            if ($this->_logger instanceof Zend_Log) {
                $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " invalid synckey $syncKey provided, invalidating sync state");
            }
            $this->_syncStateBackend->resetState($this->_device, 'FolderSync');
        } else {
            // Keep track of how many times we retried this particular sync-key
            if ($this->_syncKeyReused) {
                $extraData = json_decode($this->_syncState->extraData, true);
                $retryCounter = $extraData['retryCounter'] ?? 0;
                $retryCounter++;
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " already known synckey $syncKey provided ($retryCounter times)");
                }
                $sleepTime = 2;
                if ($retryCounter >= 10) {
                    if ($this->_logger instanceof Zend_Log) {
                        $this->_logger->warn(__METHOD__ . '::' . __LINE__ . " This client is stuck on resending the same FolderSync synckey.");
                    }
                    if (!$this->_device->isBroken) {
                        $this->_device->isBroken = true;
                        $extraData = json_decode($this->_device->extraData, true);
                        $extraData['brokenTimestamp'] = new DateTime('now', new DateTimeZone('UTC'));
                        $extraData['brokenReason'] = "This client is stuck on resending the same FolderSync synckey";
                        $this->_device->extraData = json_encode($extraData);
                        $this->_device = $this->_deviceBackend->update($this->_device); // @phpstan-ignore-line
                    }
                    call_user_func(Syncroton_Registry::getSleepCallback());
                    sleep(10);
                    header('X-MS-ASThrottle: CommandFrequency');
                    header('HTTP/1.1 503 Service Unavailable');
                    exit;
                } else {
                    $extraData['retryCounter'] = $retryCounter;
                    $this->_syncState->extraData = json_encode($extraData);
                    $this->_syncStateBackend->update($this->_syncState);
                }
                // Throttle clients that are stuck in a loop.
                // We shouldn't normally hit this codepath, so this does not have any impact on regular synchronization at all,
                // but protects us from too many requests from clients that are stuck in a loop.
                sleep($sleepTime);
            }
        }
    }

    /**
     * generate FolderSync response
     *
     * @todo changes are missing in response (folder got renamed for example)
     */
    public function getResponse()
    {
        $folderSync = $this->_outputDom->documentElement;

        // invalid synckey provided
        if (!$this->_syncState instanceof Syncroton_Model_SyncState) {
            if ($this->_logger instanceof Zend_Log) {
                $this->_logger->info(__METHOD__ . '::' . __LINE__ . " invalid synckey provided. FolderSync 0 needed.");
            }
            $folderSync->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'Status', self::STATUS_INVALID_SYNC_KEY));

            return $this->_outputDom;
        }

        // send headers from options command also when FolderSync SyncKey is 0
        if ($this->_syncState->counter == 0) {
            $optionsCommand = new Syncroton_Command_Options();
            $this->_headers = array_merge($this->_headers, $optionsCommand->getHeaders());
        }

        $adds    = [];
        $updates = [];
        $deletes = [];

        foreach ($this->_classes as $class) {
            try {
                $dataController = Syncroton_Data_Factory::factory($class, $this->_device, $this->_syncTimeStamp);
            } catch (Exception $e) {
                // backend not defined
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->info(__METHOD__ . '::' . __LINE__ . " no data backend defined for class: " . $class);
                }
                continue;
            }

            try {
                // retrieve all folders available in data backend
                $serverFolders = $dataController->getAllFolders();

                // retrieve all folders sent to client
                $clientFolders = $this->_folderBackend->getFolderState($this->_device, $class, $this->_syncState->counter);

                if ($this->_syncState->counter > 0) {
                    // retrieve all folders changed since last sync
                    $changedFolders = $dataController->getChangedFolders($this->_syncState->lastsync, $this->_syncTimeStamp);
                } else {
                    $changedFolders = [];
                }

                // only folders which were sent to the client already are allowed to be in $changedFolders
                $changedFolders = array_intersect_key($changedFolders, $clientFolders);

            } catch (Exception $e) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->crit(__METHOD__ . '::' . __LINE__ . " Syncing folder hierarchy failed: " . $e->getMessage());
                }
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " Syncing folder hierarchy failed: " . $e->getTraceAsString());
                }

                // The Status element is global for all collections. If one collection fails,
                // a failure status MUST be returned for all collections.
                if ($e instanceof Syncroton_Exception_Status) {
                    $status = $e->getCode();
                } else {
                    $status = Syncroton_Exception_Status_FolderSync::UNKNOWN_ERROR;
                }

                $folderSync->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'Status', $status));

                return $this->_outputDom;
            }

            $serverFoldersIds = array_keys($serverFolders);

            // is this the first sync?
            if ($this->_syncState->counter == 0) {
                $clientFoldersIds = [];
            } else {
                $clientFoldersIds = array_keys($clientFolders);
            }

            // calculate added entries
            $serverDiff = array_diff($serverFoldersIds, $clientFoldersIds);
            foreach ($serverDiff as $serverFolderId) {
                // have we created a folderObject in syncroton_folder before?
                if (isset($clientFolders[$serverFolderId])) {
                    $add = $clientFolders[$serverFolderId];
                    // Update some properties in case they have changed.
                    // I ran into this while changing the parentId,
                    // but the test kept returning the same result (even though I requested a fresh sync with syncKey = 0).
                    // Not sure if there are other scenarios.
                    $serverFolder = $serverFolders[$serverFolderId];
                    $add->displayName = $serverFolder->displayName;
                    $add->parentId    = $serverFolder->parentId;
                    $add->type        = $serverFolder->type;
                } else {
                    $add = $serverFolders[$serverFolderId];
                    $add->creationTime = $this->_syncTimeStamp;
                    $add->creationSynckey = $this->_syncState->counter + 1;
                    $add->deviceId     = $this->_device->id;
                    unset($add->id);
                }
                $add->class = $class;

                $adds[] = $add;
            }

            // calculate changed entries
            foreach ($changedFolders as $changedFolder) {
                $change = $clientFolders[$changedFolder->serverId];

                $change->displayName = $changedFolder->displayName;
                $change->parentId    = $changedFolder->parentId;
                $change->type        = $changedFolder->type;

                $updates[] = $change;
            }

            // Find changes in case backend does not support folder changes detection.
            // On some backends getChangedFolders() can return an empty result.
            // We make sure all is up-to-date comparing folder properties.
            foreach ($clientFoldersIds as $folderId) {
                if (isset($serverFolders[$folderId])) {
                    $c = $clientFolders[$folderId];
                    $s = $serverFolders[$folderId];

                    if ($c->displayName !== $s->displayName
                        || strval($c->parentId) !== strval($s->parentId)
                        || $c->type != $s->type
                    ) {
                        $c->displayName = $s->displayName;
                        $c->parentId    = $s->parentId;
                        $c->type        = $s->type;

                        $updates[] = $c;
                    }
                }
            }

            // Handle folders set for forced re-sync, we'll send a delete action to the client,
            // but because the folder is still existing and subscribed on the backend it should
            // "immediately" be added again (and re-synced).
            $forceDeleteIds = array_keys(array_filter($clientFolders, function ($f) { return !empty($f->resync); }));
            if (!empty($forceDeleteIds)) {
                if ($this->_logger instanceof Zend_Log) {
                    $this->_logger->debug(__METHOD__ . '::' . __LINE__ . " forcing resync of: " . var_export($forceDeleteIds, true));
                }
            }
            $serverFoldersIds = array_diff($serverFoldersIds, $forceDeleteIds);

            // calculate deleted entries
            $serverDiff = array_diff($clientFoldersIds, $serverFoldersIds);
            foreach ($serverDiff as $serverFolderId) {
                $deletes[] = $clientFolders[$serverFolderId];
            }
        }

        $folderSync->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'Status', self::STATUS_SUCCESS));

        $newSyncKey = $this->_syncState->counter;
        $count = count($adds) + count($updates) + count($deletes);
        if ($count > 0) {
            $newSyncKey++;
        }

        // create xml output
        $folderSync->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'SyncKey', $newSyncKey));

        $changes = $folderSync->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'Changes'));
        $changes->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'Count', $count));

        foreach ($adds as $folder) {
            $add = $changes->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'Add'));

            $folder->appendXML($add, $this->_device);

            if (!$this->_syncKeyReused && empty($folder->id)) {
                // The folder could exist in the backend if we e.g. delete the same name and then recreate,
                // or disable/reenable for syncing.
                if (!$this->_folderBackend->exists($this->_device->id, $folder->serverId)) {
                    $this->_folderBackend->create($folder);
                }
            }
        }

        foreach ($updates as $folder) {
            $update = $changes->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'Update'));

            $folder->appendXML($update, $this->_device);
            $this->_folderBackend->update($folder);
        }

        foreach ($deletes as $folder) {
            $delete = $changes->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'Delete'));
            $delete->appendChild($this->_outputDom->createElementNS('uri:FolderHierarchy', 'ServerId', $folder->serverId));

            $this->_folderBackend->delete($folder);
        }

        if ($this->_syncState->counter != $newSyncKey) {
            $this->_syncState->counter = $newSyncKey;
            $this->_syncState->lastsync = $this->_syncTimeStamp;
            // Keep previous sync states in case a sync key is re-sent.
            // We always insert because deleteOtherStates is executed from _syncStateBackend->validate,
            // which means we remove and re-insert the latest state on key resend.
            $this->_syncStateBackend->create($this->_syncState, true); // @phpstan-ignore-line
        }

        return $this->_outputDom;
    }
}
