<?php

/**
 * Kolab Tags backend driver for Kolab v3
 *
 * @author Aleksander Machniak <machniak@apheleia-it.ch>
 *
 * Copyright (C) 2024, Apheleia IT AG <contact@apheleia-it.ch>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

namespace KolabTags\Drivers\Kolab;

use KolabTags\Drivers\DriverInterface;
use kolab_storage_config;
use rcube_imap_generic;
use rcube_message_header;

class Driver implements DriverInterface
{
    public const O_TYPE     = 'relation';
    public const O_CATEGORY = 'tag';

    public $immutableName = false;

    private $tag_cols = ['name', 'category', 'color', 'parent', 'iconName', 'priority', 'members'];

    /**
     * Tags list
     *
     * @param array $filter Search filter
     *
     * @return array List of tags
     */
    public function list_tags($filter = [])
    {
        $config   = kolab_storage_config::get_instance();
        $default  = true;
        $filter[] = ['type', '=', self::O_TYPE];
        $filter[] = ['category', '=', self::O_CATEGORY];

        // for performance reasons assume there will be no more than 100 tags (per-folder)

        return $config->get_objects($filter, $default, 100);
    }

    /**
     * Create tag object
     *
     * @param array $tag Tag data
     *
     * @return false|array Tag data on success, False on failure
     */
    public function create($tag)
    {
        $config = kolab_storage_config::get_instance();
        $tag    = array_intersect_key($tag, array_combine($this->tag_cols, $this->tag_cols));
        $tag['category'] = self::O_CATEGORY;

        // Create the object
        $result = $config->save($tag, self::O_TYPE);

        return $result ? $tag : false;
    }

    /**
     * Update tag object
     *
     * @param array $tag Tag data
     *
     * @return false|array Tag data on success, False on failure
     */
    public function update($tag)
    {
        // get tag object data, we need _mailbox
        $list    = $this->list_tags([['uid', '=', $tag['uid']]]);
        $old_tag = $list[0] ?? null;

        if (!$old_tag) {
            return false;
        }

        $config = kolab_storage_config::get_instance();
        $tag    = array_intersect_key($tag, array_combine($this->tag_cols, $this->tag_cols));
        $tag    = array_merge($old_tag, $tag);

        // Update the object
        $result = $config->save($tag, self::O_TYPE);

        return $result ? $tag : false;
    }

    /**
     * Remove tag object
     *
     * @param string $uid Object unique identifier
     *
     * @return bool True on success, False on failure
     */
    public function remove($uid)
    {
        $config = kolab_storage_config::get_instance();

        return $config->delete($uid);
    }

    /**
     * Build IMAP SEARCH criteria for mail messages search (per-folder)
     *
     * @param array $tag     Tag data
     * @param array $folders List of folders to search in
     *
     * @return array<string> IMAP SEARCH criteria per-folder
     */
    public function members_search_criteria($tag, $folders)
    {
        $uids = kolab_storage_config::resolve_members($tag, true);
        $result  = [];

        foreach ($uids as $folder => $uid_list) {
            if (!empty($uid_list) && in_array($folder, $folders)) {
                $result[$folder] = 'UID ' . rcube_imap_generic::compressMessageSet($uid_list);
            }
        }

        return $result;
    }

    /**
     * Returns tag assignments with multiple members
     *
     * @param array<rcube_message_header> $messages Mail messages
     *
     * @return array<string, array> Tags assigned
     */
    public function members_tags($messages)
    {
        // get tags list
        $taglist = $this->list_tags();

        // get message UIDs
        $message_tags = [];
        foreach ($messages as $msg) {
            $message_tags[$msg->uid . '-' . $msg->folder] = null;
        }

        $uids = array_keys($message_tags);

        foreach ($taglist as $tag) {
            $tag['uids'] = kolab_storage_config::resolve_members($tag, true);

            foreach ((array) $tag['uids'] as $folder => $_uids) {
                array_walk($_uids, function (&$uid, $key, $folder) { $uid .= '-' . $folder; }, $folder);

                foreach (array_intersect($uids, $_uids) as $uid) {
                    $message_tags[$uid][] = $tag['uid'];
                }
            }
        }

        return array_filter($message_tags);
    }


    /**
     * Add mail members to a tag
     *
     * @param array $tag      Tag object
     * @param array $messages List of messages in rcmail::get_uids() output format
     *
     * @return bool True on success, False on error
     */
    public function add_tag_members($tag, $messages)
    {
        $storage = \rcube::get_instance()->get_storage();
        $members = [];

        // build list of members
        foreach ($messages as $mbox => $uids) {
            if ($uids === '*') {
                $index = $storage->index($mbox, null, null, true);
                $uids  = $index->get();
                $msgs  = $storage->fetch_headers($mbox, $uids, false);
            } else {
                $msgs = $storage->fetch_headers($mbox, $uids, false);
            }

            // fetch_headers doesn't detect IMAP errors, so we make sure we get something back.
            if (!empty($uids) && empty($msgs)) {
                throw new \Exception("Failed to find relation members, check the IMAP log.");
            }

            $members = array_merge($members, kolab_storage_config::build_members($mbox, $msgs));
        }

        $tag['members'] = array_unique(array_merge((array) ($tag['members'] ?? []), $members));

        // update tag object
        return $this->update($tag);
    }

    /**
     * Remove mail members from a tag
     *
     * @param array $tag      Tag object
     * @param array $messages List of messages in rcmail::get_uids() output format
     *
     * @return bool True on success, False on error
     */
    public function remove_tag_members($tag, $messages)
    {
        $filter = [];

        foreach ($messages as $mbox => $uids) {
            if ($uids === '*') {
                $filter[$mbox] = kolab_storage_config::build_member_url(['folder' => $mbox]);
            } else {
                foreach ((array)$uids as $uid) {
                    $filter[$mbox][] = kolab_storage_config::build_member_url([
                            'folder' => $mbox,
                            'uid'    => $uid,
                    ]);
                }
            }
        }

        $updated = false;

        // @todo: make sure members list is up-to-date (UIDs are up-to-date)

        // ...filter members by folder/uid prefix
        foreach ((array) $tag['members'] as $idx => $member) {
            foreach ($filter as $members) {
                // list of prefixes
                if (is_array($members)) {
                    foreach ($members as $message) {
                        if ($member == $message || strpos($member, $message . '?') === 0) {
                            unset($tag['members'][$idx]);
                            $updated = true;
                        }
                    }
                }
                // one prefix (all messages in a folder)
                else {
                    if (preg_match('/^' . preg_quote($members, '/') . '\/[0-9]+(\?|$)/', $member)) {
                        unset($tag['members'][$idx]);
                        $updated = true;
                    }
                }
            }
        }

        // update tag object
        return $updated ? $this->update($tag) : true;
    }
}
