<?php

/**
 * Kolab Tags backend driver for SQL database.
 *
 * @author Aleksander Machniak <machniak@apheleia-it.ch>
 *
 * Copyright (C) 2024, Apheleia IT AG <contact@apheleia-it.ch>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

namespace KolabTags\Drivers\Database;

use KolabTags\Drivers\DriverInterface;
use kolab_storage_cache;
use kolab_storage_config;
use rcube;
use rcube_imap_generic;
use rcube_message_header;

class Driver implements DriverInterface
{
    public $immutableName = false;

    private $members_table = 'kolab_tag_members';
    private $tags_table = 'kolab_tags';
    private $tag_cols = ['name', 'color'];


    /**
     * Tags list
     *
     * @param array $filter Search filter
     *
     * @return array List of tags
     */
    public function list_tags($filter = [])
    {
        $rcube = rcube::get_instance();
        $db = $rcube->get_dbh();
        $user_id = $rcube->get_user_id();

        // Parse filter, convert 'uid' into 'id'
        foreach ($filter as $idx => $record) {
            if ($record[0] == 'uid') {
                $filter[$idx][0] = 'id';
            }
        }

        $where = kolab_storage_cache::sql_where($filter);

        $query = $db->query("SELECT * FROM `{$this->tags_table}` WHERE `user_id` = {$user_id}" . $where);

        $result = [];
        while ($tag = $db->fetch_assoc($query)) {
            $tag['uid'] = $tag['id']; // the API expects 'uid' property
            unset($tag['id']);
            $result[] = $tag;
        }

        return $result;
    }

    /**
     * Create tag object
     *
     * @param array $tag Tag data
     *
     * @return false|array Tag data on success, False on failure
     */
    public function create($tag)
    {
        $rcube = rcube::get_instance();
        $db = $rcube->get_dbh();
        $user_id = $rcube->get_user_id();
        $insert = [];

        foreach ($this->tag_cols as $col) {
            if (isset($tag[$col])) {
                $insert[$db->quoteIdentifier($col)] = $db->quote($tag[$col]);
            }
        }

        if (empty($insert)) {
            return false;
        }

        $now = new \DateTime('now', new \DateTimeZone('UTC'));

        $insert['user_id'] = $user_id;
        $insert['created'] = $insert['updated'] = $now->format("'Y-m-d H:i:s'");

        $result = $db->query(
            "INSERT INTO `{$this->tags_table}`"
            . " (" . implode(', ', array_keys($insert)) . ")"
            . " VALUES(" . implode(', ', array_values($insert)) . ")"
        );

        $tag['uid'] = $db->insert_id($this->tags_table);

        if (empty($tag['uid'])) {
            return false;
        }

        return $tag;
    }

    /**
     * Update tag object
     *
     * @param array $tag Tag data
     *
     * @return false|array Tag data on success, False on failure
     */
    public function update($tag)
    {
        $rcube = rcube::get_instance();
        $db = $rcube->get_dbh();
        $user_id = $rcube->get_user_id();
        $update = [];

        foreach ($this->tag_cols as $col) {
            if (isset($tag[$col])) {
                $update[] = $db->quoteIdentifier($col) . ' = ' . $db->quote($tag[$col]);
            }
        }

        if (!empty($update)) {
            $now = new \DateTime('now', new \DateTimeZone('UTC'));
            $update[] = '`updated` = ' . $db->quote($now->format('Y-m-d H:i:s'));
            if (isset($tag['name'])) {
                $update[] = '`modseq` = `modseq` + (CASE WHEN name <> ' . $db->quote($tag['name']) . ' THEN 1 ELSE 0 END)';
            }

            $result = $db->query("UPDATE `{$this->tags_table}` SET " . implode(', ', $update)
                . " WHERE `id` = ? AND `user_id` = ?", $tag['uid'], $user_id);

            if ($result === false) {
                return false;
            }
        }

        return $tag;
    }

    /**
     * Remove tag object
     *
     * @param string $uid Object unique identifier
     *
     * @return bool True on success, False on failure
     */
    public function remove($uid)
    {
        $rcube = rcube::get_instance();
        $db = $rcube->get_dbh();
        $user_id = $rcube->get_user_id();

        $result = $db->query("DELETE FROM `{$this->tags_table}` WHERE `id` = ? AND `user_id` = ?", $uid, $user_id);

        return $db->affected_rows($result) > 0;
    }

    /**
     * Build IMAP SEARCH criteria for mail messages search (per-folder)
     *
     * @param array $tag     Tag data
     * @param array $folders List of folders to search in
     *
     * @return array<string> IMAP SEARCH criteria per-folder
     */
    public function members_search_criteria($tag, $folders)
    {
        $tag_members = self::resolve_members($tag);
        $result  = [];

        foreach ($tag_members as $folder => $uid_list) {
            if (!empty($uid_list) && in_array($folder, $folders)) {
                $result[$folder] = 'UID ' . rcube_imap_generic::compressMessageSet($uid_list);
            }
        }

        return $result;
    }

    /**
     * Returns tag assignments with multiple members
     *
     * @param array<rcube_message_header> $messages Mail messages
     *
     * @return array<string, array> Tags assigned
     */
    public function members_tags($messages)
    {
        // get tags list
        $taglist = $this->list_tags();

        // get message UIDs
        $message_tags = [];
        foreach ($messages as $msg) {
            $message_tags[$msg->uid . '-' . $msg->folder] = null;
        }

        $uids = array_keys($message_tags);

        foreach ($taglist as $tag) {
            $tag_members = self::resolve_members($tag);

            foreach ((array) $tag_members as $folder => $_uids) {
                array_walk($_uids, function (&$uid, $key, $folder) { $uid .= '-' . $folder; }, $folder);

                foreach (array_intersect($uids, $_uids) as $uid) {
                    $message_tags[$uid][] = $tag['uid'];
                }
            }
        }

        return array_filter($message_tags);
    }

    /**
     * Add mail members to a tag
     *
     * @param array $tag      Tag object
     * @param array $messages List of messages in rcmail::get_uids() output format
     *
     * @return bool True on success, False on error
     */
    public function add_tag_members($tag, $messages)
    {
        $rcube = rcube::get_instance();
        $storage = $rcube->get_storage();
        $members = [];

        // build list of members
        foreach ($messages as $mbox => $uids) {
            if ($uids === '*') {
                $index = $storage->index($mbox, null, null, true);
                $uids  = $index->get();
                $msgs  = $storage->fetch_headers($mbox, $uids, false);
            } else {
                $msgs = $storage->fetch_headers($mbox, $uids, false);
            }

            // fetch_headers doesn't detect IMAP errors, so we make sure we get something back.
            if (!empty($uids) && empty($msgs)) {
                throw new \Exception("Failed to find relation members, check the IMAP log.");
            }

            $members = array_merge($members, kolab_storage_config::build_members($mbox, $msgs));
        }

        if (empty($members)) {
            return false;
        }

        $db = $rcube->get_dbh();
        $existing = [];

        $query = $db->query("SELECT `url` FROM `{$this->members_table}` WHERE `tag_id` = ?", $tag['uid']);

        while ($member = $db->fetch_assoc($query)) {
            $existing[] = $member['url'];
        }

        $insert = array_unique(array_merge((array) $existing, $members));

        if (!empty($insert)) {
            $ts = (new \DateTime('now', new \DateTimeZone('UTC')))->format('Y-m-d H:i:s');

            foreach (array_chunk($insert, 100) as $chunk) {
                $query = "INSERT INTO `{$this->members_table}` (`tag_id`, `url`, `created`) VALUES ";
                foreach ($chunk as $idx => $url) {
                    $chunk[$idx] = sprintf("(%d, %s, %s)", $tag['uid'], $db->quote($url), $db->quote($ts));
                }

                $query = $db->query($query . implode(', ', $chunk));
            }
        }

        return true;
    }

    /**
     * Remove mail members from a tag
     *
     * @param array $tag      Tag object
     * @param array $messages List of messages in rcmail::get_uids() output format
     *
     * @return bool True on success, False on error
     */
    public function remove_tag_members($tag, $messages)
    {
        $rcube = rcube::get_instance();
        $db = $rcube->get_dbh();

        // TODO: Why we use URLs? It's because the original Kolab driver does this.
        // We should probably store folder, uid, and Message-ID separately (and just ignore other headers).
        // As it currently is the queries below seem to be fragile.

        foreach ($messages as $mbox => $uids) {
            if ($uids === '*') {
                $folder_url = kolab_storage_config::build_member_url(['folder' => $mbox]);
                $regexp = '^' . $folder_url . '/[0-9]+\?';
                $query = "DELETE FROM `{$this->members_table}` WHERE `tag_id` = ? AND `url` REGEXP ?";
                if ($db->query($query, $tag['uid'], $regexp) === false) {
                    return false;
                }
            } else {
                $members = [];
                foreach ((array)$uids as $uid) {
                    $members[] =  kolab_storage_config::build_member_url([
                            'folder' => $mbox,
                            'uid'    => $uid,
                    ]);
                }

                foreach (array_chunk($members, 100) as $chunk) {
                    foreach ($chunk as $idx => $member) {
                        $chunk[$idx] = 'LEFT(`url`, ' . (strlen($member) + 1) . ') = ' . $db->quote($member . '?');
                    }

                    $query = "DELETE FROM `{$this->members_table}` WHERE `tag_id` = ? AND (" . implode(' OR ', $chunk) . ")";
                    if ($db->query($query, $tag['uid']) === false) {
                        return false;

                    }
                }
            }
        }

        return true;
    }

    /**
     * Resolve tag members into (up-to-date) IMAP folder => uids map
     */
    protected function resolve_members($tag)
    {
        $rcube = rcube::get_instance();
        $db = $rcube->get_dbh();
        $existing = [];

        $query = $db->query("SELECT `url` FROM `{$this->members_table}` WHERE `tag_id` = ?", $tag['uid']);

        while ($member = $db->fetch_assoc($query)) {
            $existing[] = $member['url'];
        }

        $tag['members'] = $existing;

        // TODO: Don't force-resolve members all the time (2nd argument), store the last resolution timestamp
        // in kolab_tags table and use some interval (15 minutes?) to not do this heavy operation too often.
        $members = kolab_storage_config::resolve_members($tag, true, false);

        // Refresh the members in database
        $delete = array_unique(array_diff($existing, $tag['members']));
        $insert = array_unique(array_diff($tag['members'], $existing));

        if (!empty($delete)) {
            foreach (array_chunk($delete, 100) as $chunk) {
                $query = "DELETE FROM `{$this->members_table}` WHERE `tag_id` = ? AND `url` IN (" . $db->array2list($chunk) . ")";
                $query = $db->query($query, $tag['uid']);
            }
        }

        if (!empty($insert)) {
            $ts = (new \DateTime('now', new \DateTimeZone('UTC')))->format('Y-m-d H:i:s');

            foreach (array_chunk($insert, 100) as $chunk) {
                $query = "INSERT INTO `{$this->members_table}` (`tag_id`, `url`, `created`) VALUES ";
                foreach ($chunk as $idx => $url) {
                    $chunk[$idx] = sprintf("(%d, %s, %s)", $tag['uid'], $db->quote($url), $db->quote($ts));
                }

                $query = $db->query($query . implode(', ', $chunk));
            }
        }

        return $members;
    }
}
