<?php

/**
 * ActiveSync configuration user interface builder
 *
 * @version @package_version@
 * @author Thomas Bruederli <bruederli@kolabsys.com>
 * @author Aleksander Machniak <machniak@kolabsys.com>
 *
 * Copyright (C) 2011-2013, Kolab Systems AG <contact@kolabsys.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

class kolab_activesync_ui
{
    public $device = [];

    private $rc;
    private $plugin;
    private $skin_path;

    public const SETUP_URL = 'https://kb.kolab.org/documentation/setting-up-an-activesync-client';
    public const RESET_URL = 'https://kb.kolab.org/documentation/reset-an-activesync-client';


    public function __construct($plugin)
    {
        $this->plugin    = $plugin;
        $this->rc        = rcube::get_instance();
        $skin_path       = $this->plugin->local_skin_path() . '/';
        $this->skin_path = 'plugins/kolab_activesync/' . $skin_path;


        $this->plugin->include_stylesheet($skin_path . 'config.css');
    }

    public function device_list($attrib = [])
    {
        $attrib += ['id' => 'devices-list'];

        $devices = $this->plugin->engine->list_devices();
        $table   = new html_table();

        foreach ($devices as $id => $device) {
            $name = $device['friendlyname'] ? $device['friendlyname'] : $id;
            if ($device['is_broken']) {
                // TODO use an icon instead. Perhaps add some css styling.
                $name = "$name !";
            }
            $table->add_row(['id' => 'rcmrow' . $id]);
            $table->add(null, html::span('devicealias', rcube::Q($name))
                . ' ' . html::span('devicetype secondary', rcube::Q($device['devicetype'])));
        }

        $this->rc->output->add_gui_object('devicelist', $attrib['id']);
        $this->rc->output->set_env('devicecount', count($devices));

        $this->rc->output->include_script('list.js');

        return $table->show($attrib);
    }

    public function device_config_form($attrib = [])
    {
        $table = new html_table(['cols' => 2]);

        $field_id = 'config-device-alias';
        $input = new html_inputfield(['name' => 'devicealias', 'id' => $field_id, 'size' => 40]);
        $table->add('title', html::label($field_id, $this->plugin->gettext('devicealias')));
        $table->add(null, $input->show(!empty($this->device['friendlyname']) ? $this->device['friendlyname'] : $this->device['deviceid']));

        foreach (kolab_subscriptions::DEVICE_FIELDS as $key) {
            if ($key == 'friendlyname') {
                continue;
            }
            if ($key == 'is_broken') {
                continue;
            }

            $value = $this->device[$key] ?? null;

            if ($value) {
                $table->add('title', html::label(null, rcube::Q($this->plugin->gettext($key))));
                $table->add(null, rcube::Q($value));
            }
        }

        if ($attrib['form']) {
            $this->rc->output->add_gui_object('editform', $attrib['form']);
        }

        return $table->show($attrib);
    }

    public function folder_subscriptions($attrib = [])
    {
        if (empty($attrib['id'])) {
            $attrib['id'] = 'foldersubscriptions';
        }

        if ($this->device['is_broken']) {
            $url  = $this->rc->config->get('activesync_reset_url', self::RESET_URL);
            $vars = ['url' => $url];
            $msg  = $this->plugin->gettext(['name' => 'brokendevice', 'vars' => $vars]);
            return html::div('boxwarning', $msg);
        }

        // group folders by type (show only known types)
        $use_fieldsets = rcube_utils::get_boolean($attrib['use-fieldsets'] ?? '');
        $imei = $this->device['deviceid'];
        $html = null;

        foreach (['mail', 'contact', 'event', 'task', 'note'] as $type) {
            $subscriptions = $this->plugin->engine->list_subscriptions($imei, $type);
            $folders = [];

            foreach ($this->plugin->engine->list_folders($type) as $folder) {
                $f = $folder[0];
                $subscribed = 0;

                if (!empty($subscriptions[$f])) {
                    $subscribed = (int) $subscriptions[$f][0];
                }

                $folders[] = [$f, $folder[1], $subscribed];
            }

            if (empty($folders)) {
                continue;
            }

            $attrib['type'] = $type;
            $table = $this->folder_subscriptions_block($folders, $attrib);
            $label = $this->plugin->gettext($type);

            if ($use_fieldsets) {
                $html .= html::tag('fieldset', 'subscriptionblock', html::tag('legend', $type, $label) . $table);
            } else {
                $html .= html::div('subscriptionblock', html::tag('h3', $type, $label) . $table);
            }
        }

        $this->rc->output->add_gui_object('subscriptionslist', $attrib['id']);

        return html::div($attrib, $html);
    }

    private function folder_subscriptions_block($a_folders, $attrib)
    {
        $alarms = ($attrib['type'] == 'event' || $attrib['type'] == 'task');

        $table = new html_table(['cellspacing' => 0, 'class' => 'table-striped']);
        $table->add_header(
            [
                'class'    => 'subscription checkbox-cell',
                'title'    => $this->plugin->gettext('synchronize'),
                'tabindex' => 0,
            ],
            !empty($attrib['syncicon']) ? html::img(['src' => $this->skin_path . $attrib['syncicon']]) : $this->plugin->gettext('synchronize')
        );

        if ($alarms) {
            $table->add_header(
                [
                    'class'    => 'alarm checkbox-cell',
                    'title'    => $this->plugin->gettext('withalarms'),
                    'tabindex' => 0,
                ],
                !empty($attrib['alarmicon']) ? html::img(['src' => $this->skin_path . $attrib['alarmicon']]) : $this->plugin->gettext('withalarms')
            );
        }

        $table->add_header('foldername', $this->plugin->gettext('folder'));

        $checkbox_sync  = new html_checkbox(['name' => 'subscribed[]', 'class' => 'subscription']);
        $checkbox_alarm = new html_checkbox(['name' => 'alarm[]', 'class' => 'alarm']);

        $names = [];
        foreach ($a_folders as $folder) {
            [$folder, $foldername, $subscribed] = $folder;

            // find folder prefix to truncate (the same code as in kolab_addressbook plugin)
            $origname = $foldername;
            for ($i = count($names) - 1; $i >= 0; $i--) {
                if (strpos($foldername, $names[$i] . ' &raquo; ') === 0) {
                    $length = strlen($names[$i] . ' &raquo; ');
                    $prefix = substr($foldername, 0, $length);
                    $count  = count(explode(' &raquo; ', $prefix));
                    $foldername = str_repeat('&nbsp;&nbsp;', $count - 1) . '&raquo; ' . substr($foldername, $length);
                    break;
                }
            }

            $folder_id = 'rcmf' . rcube_utils::html_identifier($attrib['type'] . '#' . $folder);
            $folder_value = $attrib['type'] . '#' . $folder;

            $names[] = $origname;
            $classes = ['mailbox'];

            $folder_class = $attrib['type'] == 'mail' ? $this->rc->folder_classname($folder) : '';
            if ($folder_class) {
                if ($this->rc->text_exists($folder_class)) {
                    $foldername = html::quote($this->rc->gettext($folder_class));
                }
                $classes[] = $folder_class;
            }

            $table->add_row();

            $disabled = $this->plugin->engine->is_protected($folder, $this->device['devicetype'] ?? 'unknown');

            $table->add('subscription checkbox-cell', $checkbox_sync->show(
                $subscribed > 0 ? $folder_value : null,
                ['value' => $folder_value, 'id' => $folder_id, 'disabled' => $disabled]
            ));

            if ($alarms) {
                $table->add('alarm checkbox-cell', $checkbox_alarm->show(
                    $subscribed > 1 ? $folder_value : null,
                    ['value' => $folder_value, 'id' => $folder_id . '_alarm']
                ));
            }

            $table->add(implode(' ', $classes), html::label($folder_id, $foldername));
        }

        return $table->show();
    }

    /**
     * HTML table with a list of Activesync devices and an option to
     * subscribe/unsubscribe to a specified folder.
     *
     * @param string|kolab_storage_dav_folder $folder  Folder object or name
     * @param array                           $devices Devices list
     * @param string                          $type    Folder type
     *
     * @return string HTMML table
     */
    public function folder_options_table($folder, $devices, $type)
    {
        $alarms = $type == 'event' || $type == 'task';

        $table = new html_table(['cellspacing' => 0, 'id' => 'folder-sync-options', 'class' => 'records-table']);

        // table header
        $table->add_header(['class' => 'device'], $this->plugin->gettext('devicealias'));
        $table->add_header(['class' => 'subscription'], $this->plugin->gettext('synchronize'));
        if ($alarms) {
            $table->add_header(['class' => 'alarm'], $this->plugin->gettext('withalarms'));
        }

        $folder_name = is_object($folder) ? $folder->href : $folder;
        $subscriptions = $this->plugin->engine->folder_subscriptions($folder_name, $type);

        // table records
        foreach ($devices as $id => $device) {
            $name = $id;
            $title = '';

            $_name = trim(($device['friendlyname'] ?? '') . ' ' . ($device['os'] ?? ''));
            $title = $device['useragent'] ?? '';

            if ($_name) {
                $name .= " ($_name)";
            }

            $disabled = $this->plugin->engine->is_protected($folder_name, $device['devicetype'] ?? 'unknown');
            $flag = $subscriptions[$id] ?? 0;

            $table->add_row();
            $table->add(['class' => 'device', 'title' => $title], $name);

            $checkbox = new html_checkbox(['name' => "_subscriptions[$id]", 'value' => 1,
                'onchange' => 'return activesync_object.update_sync_data(this)']);

            $table->add('subscription checkbox-cell', $checkbox->show($flag ? 1 : 0, ['disabled' => $disabled]));

            if ($alarms) {
                $checkbox_alarm = new html_checkbox(['name' => "_alarms[$id]", 'value' => 1,
                    'onchange' => 'return activesync_object.update_sync_data(this)']);

                $table->add('alarm checkbox-cell', $checkbox_alarm->show($flag > 1 ? 1 : 0));
            }
        }

        return $table->show();
    }

    /**
     * Displays initial page (when no devices are registered)
     */
    public function init_message()
    {
        $this->rc->output->add_handlers([
                'initmessage' => [$this, 'init_message_content'],
        ]);

        $this->rc->output->send('kolab_activesync.configempty');
    }

    /**
     * Handler for initmessage template object
     */
    public function init_message_content()
    {
        $url  = $this->rc->config->get('activesync_setup_url', self::SETUP_URL);
        $vars = ['url' => $url];
        $msg  = $this->plugin->gettext(['name' => 'nodevices', 'vars' => $vars]);

        return $msg;
    }
}
