<?php

/**
 * ActiveSync configuration utility for Kolab accounts
 *
 * @version @package_version@
 * @author Aleksander Machniak <machniak@kolabsys.com>
 * @author Thomas Bruederli <bruederli@kolabsys.com>
 *
 * Copyright (C) 2024, Apheleia IT AG <contact@apheleia-it.ch>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

class kolab_activesync extends rcube_plugin
{
    /** @var string Defines when the plugin is being used */
    public $task = '(addressbook|calendar|settings|tasks)';

    /** @var ?kolab_subscriptions Subscriptions storage */
    public $engine;

    private $rc;
    private $ui;

    /**
     * Plugin initialization.
     */
    public function init()
    {
        $this->rc = rcmail::get_instance();

        $this->require_plugin('libkolab');

        $this->register_action('plugin.activesync-json', [$this, 'json_command']);

        if ($this->rc->task == 'settings') {
            $this->register_action('plugin.activesync', [$this, 'config_view']);
            $this->register_action('plugin.activesync-config', [$this, 'config_frame']);

            $this->add_hook('settings_actions', [$this, 'settings_actions']);
            $this->add_hook('folder_form', [$this, 'folder_form']);
        } else {
            $this->add_hook('kolab_folder_form', [$this, 'folder_form']);
        }
    }

    /**
     * Adds Activesync section in Settings
     */
    public function settings_actions($args)
    {
        $this->init_ui();

        $args['actions'][] = [
            'action' => 'plugin.activesync',
            'class'  => 'activesync',
            'label'  => 'tabtitle',
            'domain' => 'kolab_activesync',
            'title'  => 'activesynctitle',
        ];

        return $args;
    }

    /**
     * Handler for folder info/edit form (folder_form hook).
     * Adds ActiveSync section.
     */
    public function folder_form($args)
    {
        // Note: 'folder' arg from kolab_folder_form, 'options' arg from kolab_form hook
        $folder = $args['folder'] ?? ($args['options']['name'] ?? null);

        // Edited folder name (empty in create-folder mode)
        if ($folder === null || $folder === '') {
            return $args;
        }

        $this->init_engine();

        $devices = $this->engine->list_devices();

        // no registered devices
        if (empty($devices)) {
            return $args;
        }

        if (is_object($folder)) {
            $type = $folder->type;
        } else {
            [$type, ] = explode('.', (string) kolab_storage::folder_type($folder));
        }

        if ($type && !in_array($type, ['mail', 'event', 'contact', 'task', 'note'])) {
            return $args;
        }

        $this->init_ui();

        if ($content = $this->ui->folder_options_table($folder, $devices, $type)) {
            $args['form']['activesync'] = [
                'name'    => $this->gettext('tabtitle'),
                'content' => $content,
            ];

            $this->add_label('savingdata');
            $this->include_script('kolab_activesync.js');

            $this->rc->output->set_env('activesync_folder', is_object($folder) ? $folder->href : $folder);
            $this->rc->output->set_env('activesync_type', $type);
        }

        return $args;
    }

    /**
     * Handle JSON requests
     */
    public function json_command()
    {
        $cmd  = rcube_utils::get_input_value('cmd', rcube_utils::INPUT_POST);
        $imei = rcube_utils::get_input_value('id', rcube_utils::INPUT_POST);

        $this->init_engine();
        $this->add_texts('localization/');

        switch ($cmd) {
            case 'save':
                $subscriptions = (array) rcube_utils::get_input_value('subscribed', rcube_utils::INPUT_POST);
                $devicealias = rcube_utils::get_input_value('devicealias', rcube_utils::INPUT_POST, true);

                $err = !$this->engine->device_update($imei, ['friendlyname' => $devicealias]);

                if (!$err) {
                    $subs = ['mail' => [], 'contact' => [], 'event' => [], 'task' => [], 'note' => []];
                    $regexp = '/^(' . implode('|', array_keys($subs)) . ')#(.*)$/';

                    foreach ($subscriptions as $folder => $flag) {
                        if (empty($flag)) {
                            continue;
                        }

                        if (preg_match($regexp, $folder, $m)) {
                            $type = $m[1];
                            $folder = $m[2];

                            $subs[$type][$folder] = (int) $flag;
                        }
                    }

                    foreach ($subs as $type => $list) {
                        $err = !$this->engine->set_subscriptions($imei, $type, $list);
                        if ($err) {
                            break;
                        }
                    }

                    $this->rc->output->command('plugin.activesync_save_complete', [
                        'success' => !$err, 'id' => $imei, 'alias' => rcube::Q($devicealias)]);
                }

                if ($err) {
                    $this->rc->output->show_message($this->gettext('savingerror'), 'error');
                } else {
                    $this->rc->output->show_message($this->gettext('successfullysaved'), 'confirmation');
                }

                break;

            case 'delete':
                foreach ((array) $imei as $id) {
                    $success = $this->engine->device_delete($id);
                }

                if (!empty($success)) {
                    $this->rc->output->show_message($this->gettext('successfullydeleted'), 'confirmation');
                    $this->rc->output->command('plugin.activesync_save_complete', [
                            'success' => true,
                            'delete'  => true,
                            'id'      => count($imei) > 1 ? 'ALL' : $imei[0],
                    ]);
                } else {
                    $this->rc->output->show_message($this->gettext('savingerror'), 'error');
                }

                break;

            case 'update':
                $flag = (int) rcube_utils::get_input_value('flag', rcube_utils::INPUT_POST);
                $folder = rcube_utils::get_input_value('folder', rcube_utils::INPUT_POST);
                $type = rcube_utils::get_input_value('type', rcube_utils::INPUT_POST);

                $err = !$this->engine->folder_subscribe($imei, $folder, $flag, $type);

                if ($err) {
                    $this->rc->output->show_message($this->gettext('savingerror'), 'error');
                } else {
                    $this->rc->output->show_message($this->gettext('successfullysaved'), 'confirmation');
                }

                break;
        }

        $this->rc->output->send();
    }

    /**
     * Render main UI for devices configuration
     */
    public function config_view()
    {
        $this->init_ui();

        $this->register_handler('plugin.devicelist', [$this->ui, 'device_list']);

        $this->add_label('devicedeleteconfirm', 'savingdata');
        $this->include_script('kolab_activesync.js');

        $this->rc->output->send('kolab_activesync.config');
    }

    /**
     * Render device configuration form
     */
    public function config_frame()
    {
        $this->init_ui();

        if (!empty($_GET['_init'])) {
            return $this->ui->init_message();
        }

        $this->include_script('kolab_activesync.js');

        $this->register_handler('plugin.deviceconfigform', [$this->ui, 'device_config_form']);
        $this->register_handler('plugin.foldersubscriptions', [$this->ui, 'folder_subscriptions']);

        $imei = rcube_utils::get_input_value('_id', rcube_utils::INPUT_GPC);

        if ($device = $this->engine->device_info($imei)) {
            $this->ui->device = $device;
            $this->rc->output->set_env('active_device', $imei);
            $this->rc->output->command('parent.enable_command', 'plugin.delete-device', true);
        } else {
            $this->rc->output->show_message($this->gettext('devicenotfound'), 'error');
        }

        $this->rc->output->send('kolab_activesync.configedit');
    }

    /**
     * Initialize the subscriptions engine
     */
    private function init_engine()
    {
        if (!$this->engine) {
            $this->load_config();
            $this->engine = new kolab_subscriptions($this->rc->config->get('activesync_dav_server'));
        }
    }

    /**
     * Initialize the UI engine
     */
    private function init_ui()
    {
        if (!$this->ui) {
            $this->init_engine();
            require_once $this->home . '/kolab_activesync_ui.php';
            $this->ui = new kolab_activesync_ui($this);

            $this->add_texts('localization/');
        }
    }
}
