<?php

class PingTest extends Tests\SyncTestCase
{
    /**
     * Test Ping command
     */
    public function testFolderSyncBasic()
    {
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Ping xmlns="uri:Ping">
                <HeartbeatInterval>900</HeartbeatInterval>
                <Folders>
                    <Folder>
                        <Id>38b950ebd62cd9a66929c89615d0fc04</Id>
                        <Class>Email</Class>
                    </Folder>
                </Folders>
            </Ping>
            EOF;

        $response = $this->request($request, 'Ping');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        // Initially we know no folders
        $this->assertSame('7', $xpath->query("//ns:Ping/ns:Status")->item(0)->nodeValue);


        // We discover folders with a foldersync
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());

        // Now we get to the actual ping
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Ping xmlns="uri:Ping">
                <HeartbeatInterval>0</HeartbeatInterval>
                <Folders>
                    <Folder>
                        <Id>38b950ebd62cd9a66929c89615d0fc04</Id>
                        <Class>Email</Class>
                    </Folder>
                </Folders>
            </Ping>
            EOF;

        $response = $this->request($request, 'Ping');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        // We don't find the folder state because it was never synced
        $this->assertSame('2', $xpath->query("//ns:Ping/ns:Status")->item(0)->nodeValue);

        // Sync inbox
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Sync xmlns="uri:AirSync" xmlns:AirSyncBase="uri:AirSyncBase">
                <Collections>
                    <Collection>
                        <SyncKey>0</SyncKey>
                        <CollectionId>38b950ebd62cd9a66929c89615d0fc04</CollectionId>
                        <DeletesAsMoves>1</DeletesAsMoves>
                        <GetChanges>1</GetChanges>
                        <Options>
                            <FilterType>0</FilterType>
                            <Conflict>1</Conflict>
                            <BodyPreference xmlns="uri:AirSyncBase">
                                <Type>2</Type>
                                <TruncationSize>51200</TruncationSize>
                                <AllOrNone>0</AllOrNone>
                            </BodyPreference>
                        </Options>
                    </Collection>
                </Collections>
            </Sync>
            EOF;

        $response = $this->request($request, 'Sync');
        $this->assertEquals(200, $response->getStatusCode());

        // Now we should get no change
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Ping xmlns="uri:Ping">
                <HeartbeatInterval>0</HeartbeatInterval>
                <Folders>
                    <Folder>
                        <Id>38b950ebd62cd9a66929c89615d0fc04</Id>
                        <Class>Email</Class>
                    </Folder>
                </Folders>
            </Ping>
            EOF;

        $response = $this->request($request, 'Ping');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:Ping/ns:Status")->item(0)->nodeValue);
    }

    /**
     * Test Unknown Folder
     */
    public function testUnknownFolder()
    {
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Ping xmlns="uri:Ping">
                <HeartbeatInterval>900</HeartbeatInterval>
                <Folders>
                    <Folder>
                        <Id>foobar</Id>
                        <Class>Email</Class>
                    </Folder>
                </Folders>
            </Ping>
            EOF;

        $response = $this->request($request, 'Ping');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('7', $xpath->query("//ns:Ping/ns:Status")->item(0)->nodeValue);
    }

    /**
     * Test New Folder
     */
    public function testNewFolder()
    {
        // Initialize the folder state
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());

        $this->createTestFolder("NewFolder", "mail");
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Ping xmlns="uri:Ping">
                <HeartbeatInterval>900</HeartbeatInterval>
                <Folders>
                    <Folder>
                        <Id>38b950ebd62cd9a66929c89615d0fc04</Id>
                        <Class>Email</Class>
                    </Folder>
                </Folders>
            </Ping>
            EOF;

        $response = $this->request($request, 'Ping');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('7', $xpath->query("//ns:Ping/ns:Status")->item(0)->nodeValue);
    }

    /**
     * Test Changed Subscription Folder
     */
    public function testNewFolderSubscriptionState()
    {
        // Initialize the folder state
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());

        $this->setSubscriptionState("NewFolder", 'mail', null);
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Ping xmlns="uri:Ping">
                <HeartbeatInterval>900</HeartbeatInterval>
                <Folders>
                    <Folder>
                        <Id>38b950ebd62cd9a66929c89615d0fc04</Id>
                        <Class>Email</Class>
                    </Folder>
                </Folders>
            </Ping>
            EOF;

        $response = $this->request($request, 'Ping');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('7', $xpath->query("//ns:Ping/ns:Status")->item(0)->nodeValue);
    }

    /**
     * Test changed subscription while ping is running
     */
    public function testNewFolderSubscriptionStateDuringPing()
    {
        // Initialize the folder state
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());

        // Set the metadata via a background script, while Ping is running.
        // This is essential to trigger a bug where the initial Ping seeds the roundcube metadata cache,
        // and then subsequent checks don't update the cache. We have to use a separate process to set
        // the metadata, because otherwise we update the cache (we could also not use the roundcube
        // infrastructure instead).
        $deviceId = self::$deviceId;
        $username = self::$username;
        $password = self::$password;
        $script = TESTS_DIR . 'bootstrap.php';
        exec("php {$script} subscribe --folder=NewFolder --type=mail --delay=5"
            . " --device={$deviceId} --user={$username} --password={$password} &", $output);
        $this->assertSame([], $output);

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <Ping xmlns="uri:Ping">
                <HeartbeatInterval>9999</HeartbeatInterval>
                <Folders>
                    <Folder>
                        <Id>38b950ebd62cd9a66929c89615d0fc04</Id>
                        <Class>Email</Class>
                    </Folder>
                </Folders>
            </Ping>
            EOF;

        $response = $this->request($request, 'Ping');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('7', $xpath->query("//ns:Ping/ns:Status")->item(0)->nodeValue);
    }
}
