<?php

class FoldersTest extends Tests\SyncTestCase
{
    /**
     * Cleanup folders
     */
    protected function foldersCleanup(): void
    {
        // Note: We essentially assume the test account is in an initial state, extra folders may break tests
        // Anyway, we first remove folders that might have been created during tests in this file
        $this->deleteTestFolder('NewFolder', 'mail');
        $this->deleteTestFolder('NewFolder2', 'mail');
        $this->deleteTestFolder('Test Folder', 'mail');
        $this->deleteTestFolder('Test Folder New', 'mail');
        $this->deleteTestFolder('Test Contacts Folder', 'contact');
        $this->deleteTestFolder('Test Contacts New', 'contact');
        // Make sure the default folders exist
        if (!$this->isStorageDriver('kolab4')) {
            $this->createTestFolder('Calendar', 'event.default');
            $this->createTestFolder('Contacts', 'contact.default');
            $this->createTestFolder('Tasks', 'task.default');
            $this->createTestFolder('Notes', 'note.default');
        }
        //TODO: handle kolab4 case?
    }

    /**
     * Test FolderSync command
     */
    public function testFolderSyncBasic()
    {
        $this->registerDevice();
        $this->foldersCleanup();

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        // We expect some folders to exist (dont' know how many)
        $this->assertTrue(intval($xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue) > 2);

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        // No changes on second sync
        $this->assertSame(strval(0), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        //Clear the creation_synckey (that's the migration scenario)
        //Shouldn't trigger a change
        $rcube = \rcube::get_instance();
        $db    = $rcube->get_dbh();
        $result = $db->query(
            "UPDATE `syncroton_folder` SET `creation_synckey` = 0",
        );

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        // No changes on second sync
        $this->assertSame(strval(0), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);
    }

    /**
     * Test invalid sync key
     */
    public function testFolderInvalidSyncKey()
    {
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>999</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('9', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
    }

    /**
     * Test synckey reuse
     */
    public function testSyncKeyResend()
    {
        $this->createTestFolder("NewFolderToRemove", "mail");
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;
        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);

        //Now change something
        $this->createTestFolder("NewFolder", "mail");
        $this->deleteTestFolder('NewFolderToRemove', 'mail');

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('2', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(strval(2), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        //Resend the same synckey
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('2', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(strval(2), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        //And now make sure we can still move on
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>2</SyncKey>
            </FolderSync>
            EOF;
        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('2', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);

        //Now add another folder
        $this->createTestFolder("NewFolder2", "mail");
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>2</SyncKey>
            </FolderSync>
            EOF;
        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('3', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(strval(1), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        //And finally make sure we can't go back two synckeys (because that has been cleaned up meanwhile)
        $this->createTestFolder("NewFolder2", "mail");
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
            </FolderSync>
            EOF;
        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('9', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
    }

    /**
     * This test recreates a previously deleted folder.
     * Currently similar to disabling/reenabling a folder for sync, but should perhaps be tested separately
     *
     * @depends testSyncKeyResend
     */
    public function testRecreatePreviousFolder()
    {
        $this->deleteTestFolder('NewFolder', 'mail');
        $this->deleteTestFolder('NewFolder2', 'mail');
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;
        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);

        //Now change something
        $this->createTestFolder("NewFolder", "mail");
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('2', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(strval(1), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        // Cleanup for the other tests
        $this->deleteTestFolder('NewFolder', 'mail');
        $this->deleteTestFolder('NewFolder2', 'mail');
    }

    /**
     * Test FolderSync command
     */
    public function testFolderSync()
    {
        $this->foldersCleanup();
        $this->resetDevice();

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        // Note: We're expecting activesync_init_subscriptions=0 here.
        if ($this->isStorageDriver('kolab4')) {
            $folders = [
                ['Calendar', Syncroton_Command_FolderSync::FOLDERTYPE_CALENDAR],
                ['Contacts', Syncroton_Command_FolderSync::FOLDERTYPE_CONTACT],
                ['INBOX', Syncroton_Command_FolderSync::FOLDERTYPE_INBOX],
                ['Drafts', Syncroton_Command_FolderSync::FOLDERTYPE_DRAFTS],
                ['Sent', Syncroton_Command_FolderSync::FOLDERTYPE_SENTMAIL],
                ['Trash', Syncroton_Command_FolderSync::FOLDERTYPE_DELETEDITEMS],
                ['Spam', Syncroton_Command_FolderSync::FOLDERTYPE_MAIL_USER_CREATED],
                ['Tasks', Syncroton_Command_FolderSync::FOLDERTYPE_TASK],
            ];

        } else {
            $folders = [
                ['Calendar', Syncroton_Command_FolderSync::FOLDERTYPE_CALENDAR],
                ['Contacts', Syncroton_Command_FolderSync::FOLDERTYPE_CONTACT],
                ['INBOX', Syncroton_Command_FolderSync::FOLDERTYPE_INBOX],
                ['Drafts', Syncroton_Command_FolderSync::FOLDERTYPE_DRAFTS],
                ['Sent', Syncroton_Command_FolderSync::FOLDERTYPE_SENTMAIL],
                ['Trash', Syncroton_Command_FolderSync::FOLDERTYPE_DELETEDITEMS],
                ['Notes', Syncroton_Command_FolderSync::FOLDERTYPE_NOTE],
                ['Tasks', Syncroton_Command_FolderSync::FOLDERTYPE_TASK],
            ];
        }

        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(strval(count($folders)), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        foreach ($folders as $idx => $folder) {
            $this->assertSame($folder[0], $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:DisplayName")->item($idx)->nodeValue);
            $this->assertSame((string) $folder[1], $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:Type")->item($idx)->nodeValue);
            $this->assertSame('0', $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:ParentId")->item($idx)->nodeValue);
        }

        // Test with multi-folder support enabled
        self::$deviceType = 'iphone';

        $response = $this->request($request, 'FolderSync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        if ($this->isStorageDriver('kolab4')) {
            $folders = [
                ['Calendar', Syncroton_Command_FolderSync::FOLDERTYPE_CALENDAR],
                // Note: Kolab 4 with Cyrus DAV uses Addressbook, but Kolab 3 with iRony would use 'Contacts'
                ['/^(Contacts|Addressbook)$/', Syncroton_Command_FolderSync::FOLDERTYPE_CONTACT],
                ['INBOX', Syncroton_Command_FolderSync::FOLDERTYPE_INBOX],
                ['Drafts', Syncroton_Command_FolderSync::FOLDERTYPE_DRAFTS],
                ['Sent', Syncroton_Command_FolderSync::FOLDERTYPE_SENTMAIL],
                ['Trash', Syncroton_Command_FolderSync::FOLDERTYPE_DELETEDITEMS],
                // Note: For now Kolab 4 uses the same Calendar folder for calendar and tasks
                ['/^(Tasks|Calendar)$/', Syncroton_Command_FolderSync::FOLDERTYPE_TASK],
            ];
        }

        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(strval(count($folders)), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        foreach ($folders as $idx => $folder) {
            $displayName = $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:DisplayName")->item($idx)->nodeValue;
            if (str_starts_with($folder[0], '/')) {
                $this->assertMatchesRegularExpression($folder[0], $displayName);
            } else {
                $this->assertSame($folder[0], $displayName);
            }
            $this->assertSame((string) $folder[1], $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:Type")->item($idx)->nodeValue);
            $this->assertSame('0', $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:ParentId")->item($idx)->nodeValue);
            $idx++;
        }

        // After we switched to multi-folder supported mode we expect next FolderSync
        // to delete the old "collective" folders
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>1</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        // This depends on multifolder blacklists to be configured (otherwise we get no "collective" folders)
        $deleted = $this->isStorageDriver('kolab4') ? 4 : 5; // No Notes folder in Kolab4
        $syncKey = 2;

        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval($syncKey), $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(strval($deleted), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);
        $this->assertSame($deleted, $xpath->query("//ns:FolderSync/ns:Changes/ns:Delete")->length);

        return $syncKey;
    }

    /**
     * Test FolderCreate command
     *
     * @depends testFolderSync
     */
    public function testFolderCreate($syncKey)
    {
        // Multi-folder mode
        self::$deviceType = 'iphone';

        // Create a mail folder
        $folderName1 = 'Test Folder';
        $folderType = Syncroton_Command_FolderSync::FOLDERTYPE_MAIL_USER_CREATED;
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderCreate xmlns="uri:FolderHierarchy">
                <SyncKey>{$syncKey}</SyncKey>
                <ParentId>0</ParentId>
                <DisplayName>{$folderName1}</DisplayName>
                <Type>{$folderType}</Type>
            </FolderCreate>
            EOF;

        $response = $this->request($request, 'FolderCreate');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderCreate/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("//ns:FolderCreate/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(1, $xpath->query("//ns:FolderCreate/ns:ServerId")->count());
        $folder1 = $xpath->query("//ns:FolderCreate/ns:ServerId")->item(0)->nodeValue;

        // Note: After FolderCreate there are no changes in the following FolderSync expected

        // Create a contacts folder
        $folderName2 = 'Test Contacts Folder';
        $folderType = Syncroton_Command_FolderSync::FOLDERTYPE_CONTACT_USER_CREATED;
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderCreate xmlns="uri:FolderHierarchy">
                <SyncKey>{$syncKey}</SyncKey>
                <ParentId>0</ParentId>
                <DisplayName>{$folderName2}</DisplayName>
                <Type>{$folderType}</Type>
            </FolderCreate>
            EOF;

        $response = $this->request($request, 'FolderCreate');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderCreate/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$syncKey), $xpath->query("//ns:FolderCreate/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(1, $xpath->query("//ns:FolderCreate/ns:ServerId")->count());
        $folder2 = $xpath->query("//ns:FolderCreate/ns:ServerId")->item(0)->nodeValue;

        // Note: After FolderCreate there are no changes in the following FolderSync expected

        // TODO: Test folder with a parent

        return [
            'SyncKey' => $syncKey,
            'folders' => [
                $folder1,
                $folder2,
            ],
        ];
    }

    /**
     * Test FolderUpdate command
     *
     * @depends testFolderCreate
     */
    public function testFolderUpdate($params)
    {
        // Multi-folder mode
        self::$deviceType = 'iphone';

        // Test renaming a mail folder
        $folderType = Syncroton_Command_FolderSync::FOLDERTYPE_MAIL_USER_CREATED;
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderUpdate xmlns="uri:FolderHierarchy">
                <SyncKey>{$params['SyncKey']}</SyncKey>
                <ServerId>{$params['folders'][0]}</ServerId>
                <ParentId/>
                <DisplayName>Test Folder New</DisplayName>
                <Type>{$folderType}</Type>
            </FolderUpdate>
            EOF;

        $response = $this->request($request, 'FolderUpdate');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderUpdate/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$params['SyncKey']), $xpath->query("//ns:FolderUpdate/ns:SyncKey")->item(0)->nodeValue);

        // Test FolderSync after folder update, get the new folder id (for delete test)
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>{$params['SyncKey']}</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        // Note we expect Add+Delete here, instead of Update (but this could change in the future)
        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$params['SyncKey']), $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);
        $this->assertSame(1, $xpath->query("//ns:FolderSync/ns:Changes/ns:Add")->length);
        $this->assertSame(1, $xpath->query("//ns:FolderSync/ns:Changes/ns:Delete")->length);
        $this->assertSame($params['folders'][0], $xpath->query("//ns:FolderSync/ns:Changes/ns:Delete/ns:ServerId")->item(0)->nodeValue);
        $this->assertSame('0', $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:ParentId")->item(0)->nodeValue);
        $this->assertSame('Test Folder New', $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:DisplayName")->item(0)->nodeValue);
        $this->assertSame(strval($folderType), $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:Type")->item(0)->nodeValue);
        $params['folders'][0] = $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:ServerId")->item(0)->nodeValue;

        // Test renaming a contacts folder
        $folderType = Syncroton_Command_FolderSync::FOLDERTYPE_CONTACT_USER_CREATED;
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderUpdate xmlns="uri:FolderHierarchy">
                <SyncKey>{$params['SyncKey']}</SyncKey>
                <ServerId>{$params['folders'][1]}</ServerId>
                <ParentId/>
                <DisplayName>Test Contacts New</DisplayName>
                <Type>{$folderType}</Type>
            </FolderUpdate>
            EOF;

        $response = $this->request($request, 'FolderUpdate');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderUpdate/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$params['SyncKey']), $xpath->query("//ns:FolderUpdate/ns:SyncKey")->item(0)->nodeValue);

        // Test FolderSync after folder update, get the new folder id (for delete test)
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>{$params['SyncKey']}</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$params['SyncKey']), $xpath->query("//ns:FolderSync/ns:SyncKey")->item(0)->nodeValue);

        if ($this->isStorageDriver('kolab4')) {
            // Note we expect Update here, not Add+Delete, folder ID does not change
            $this->assertSame('1', $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);
            $this->assertSame($params['folders'][1], $xpath->query("//ns:FolderSync/ns:Changes/ns:Update/ns:ServerId")->item(0)->nodeValue);
            $this->assertSame('Test Contacts New', $xpath->query("//ns:FolderSync/ns:Changes/ns:Update/ns:DisplayName")->item(0)->nodeValue);
            $this->assertSame(strval($folderType), $xpath->query("//ns:FolderSync/ns:Changes/ns:Update/ns:Type")->item(0)->nodeValue);
        } else {
            // Note we expect Add+Delete here, instead of Update (but this could change in the future)
            $this->assertSame('2', $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);
            $this->assertSame($params['folders'][1], $xpath->query("//ns:FolderSync/ns:Changes/ns:Delete/ns:ServerId")->item(0)->nodeValue);
            $this->assertSame('0', $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:ParentId")->item(0)->nodeValue);
            $this->assertSame('Test Contacts New', $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:DisplayName")->item(0)->nodeValue);
            $this->assertSame(strval($folderType), $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:Type")->item(0)->nodeValue);
            $params['folders'][1] = $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:ServerId")->item(0)->nodeValue;
        }

        // TODO: Test folder with a parent change
        // TODO: Assert the folder name has changed in the storage
        // TODO: Test Sync after a DAV folder rename made in another client

        return $params;
    }

    /**
     * Test FolderDelete command
     *
     * @depends testFolderUpdate
     */
    public function testFolderDelete($params)
    {
        // Multi-folder mode
        self::$deviceType = 'iphone';

        // Delete mail folder
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderDelete xmlns="uri:FolderHierarchy">
                <SyncKey>{$params['SyncKey']}</SyncKey>
                <ServerId>{$params['folders'][0]}</ServerId>
            </FolderDelete>
            EOF;

        $response = $this->request($request, 'FolderDelete');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderDelete/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$params['SyncKey']), $xpath->query("//ns:FolderDelete/ns:SyncKey")->item(0)->nodeValue);

        // Note: After FolderDelete there are no changes in the following FolderSync expected

        // Delete contacts folder
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderDelete xmlns="uri:FolderHierarchy">
                <SyncKey>{$params['SyncKey']}</SyncKey>
                <ServerId>{$params['folders'][1]}</ServerId>
            </FolderDelete>
            EOF;

        $response = $this->request($request, 'FolderDelete');

        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $this->assertSame('1', $xpath->query("//ns:FolderDelete/ns:Status")->item(0)->nodeValue);
        $this->assertSame(strval(++$params['SyncKey']), $xpath->query("//ns:FolderDelete/ns:SyncKey")->item(0)->nodeValue);

        // Note: After FolderDelete there are no changes in the following FolderSync expected

        // TODO: Assert the folders no longer exist
    }


    /**
     * Test subscriptions
     */
    public function testSubscriptions()
    {
        $this->foldersCleanup();

        // Should show up although not subscribed in IMAP
        $this->createTestFolder('IMAPUnsubscribed', 'mail', '1', false);

        // Shouldn't show up
        $this->createTestFolder('ActivesyncUnsubscribed', 'mail', '0', true);

        // Should show up
        $this->createTestFolder('ActivesyncSubscribed', 'mail', '1', true);

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $folderNames = [];
        foreach ($xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:DisplayName") as $item) {
            $folderNames[] = $item->nodeValue;
        }
        $this->assertContains('IMAPUnsubscribed', $folderNames);
        $this->assertContains('ActivesyncSubscribed', $folderNames);
        $this->assertNotContains('ActivesyncUnsubscribed', $folderNames);

        // Cleanup for the other tests
        $this->deleteTestFolder('IMAPUnsubscribed', 'mail');
        $this->deleteTestFolder('ActivesyncSubscribed', 'mail');
        $this->deleteTestFolder('ActivesyncUnsubscribed', 'mail');
    }

    /**
     * Test shared folder subscription
     */
    public function testSharedFolder()
    {
        if (empty(self::$secondary_username)) {
            $this->markTestSkipped("No secondary test user configured");
        }
        $this->foldersCleanup();
        $this->createSharedTestFolder(self::$secondary_username, 'TestSharedFolder', 'mail');
        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $localPart = explode('@', self::$secondary_username)[0];
        $folders = [
            ['Calendar', Syncroton_Command_FolderSync::FOLDERTYPE_CALENDAR],
            ['/^(Contacts|Addressbook)$/', Syncroton_Command_FolderSync::FOLDERTYPE_CONTACT],
            ['INBOX', Syncroton_Command_FolderSync::FOLDERTYPE_INBOX],
            ['Drafts', Syncroton_Command_FolderSync::FOLDERTYPE_DRAFTS],
            ['Sent', Syncroton_Command_FolderSync::FOLDERTYPE_SENTMAIL],
            ['Trash', Syncroton_Command_FolderSync::FOLDERTYPE_DELETEDITEMS],
            ["($localPart) TestSharedFolder", Syncroton_Command_FolderSync::FOLDERTYPE_MAIL_USER_CREATED],
            ['Spam', Syncroton_Command_FolderSync::FOLDERTYPE_MAIL_USER_CREATED],
            ['Tasks', Syncroton_Command_FolderSync::FOLDERTYPE_TASK],
        ];

        $this->assertSame(strval(count($folders)), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        foreach ($folders as $idx => $folder) {
            $displayName = $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:DisplayName")->item($idx)->nodeValue;
            if (str_starts_with($folder[0], '/')) {
                $this->assertMatchesRegularExpression($folder[0], $displayName);
            } else {
                $this->assertSame($folder[0], $displayName);
            }

            $this->assertSame((string) $folder[1], $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:Type")->item($idx)->nodeValue);
            $this->assertSame('0', $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:ParentId")->item($idx)->nodeValue);
        }
    }

    /**
     * Test subscriptions
     */
    // Expects: $config['activesync_force_subscriptions'] = ['windowsoutlook15' => ['INBOX' => 1, 'Sent' => 1, 'Trash' => 1, 'Calendar' => 1, 'Contacts' => 1, 'Addressbook' => 1, 'Tasks' => 1, '/dav/calendars/user/.*/Default' => 1, '/dav/addressbooks/user/.*/Default' => 1]];
    public function testCleanSubscriptionState()
    {
        $this->foldersCleanup();
        $this->wipeSubscriptionState('mail');
        $this->wipeSubscriptionState('event');
        $this->wipeSubscriptionState('contact');
        $this->wipeSubscriptionState('task');
        $this->wipeSubscriptionState('note');

        $request = <<<EOF
            <?xml version="1.0" encoding="utf-8"?>
            <!DOCTYPE AirSync PUBLIC "-//AIRSYNC//DTD AirSync//EN" "http://www.microsoft.com/">
            <FolderSync xmlns="uri:FolderHierarchy">
                <SyncKey>0</SyncKey>
            </FolderSync>
            EOF;

        $response = $this->request($request, 'FolderSync');
        $this->assertEquals(200, $response->getStatusCode());

        $dom = $this->fromWbxml($response->getBody());
        $xpath = $this->xpath($dom);

        $folders = [
            ['Calendar', Syncroton_Command_FolderSync::FOLDERTYPE_CALENDAR],
            ['/^(Contacts|Addressbook)$/', Syncroton_Command_FolderSync::FOLDERTYPE_CONTACT],
            ['INBOX', Syncroton_Command_FolderSync::FOLDERTYPE_INBOX],
            ['Sent', Syncroton_Command_FolderSync::FOLDERTYPE_SENTMAIL],
            ['Spam', Syncroton_Command_FolderSync::FOLDERTYPE_MAIL_USER_CREATED],
            ['Trash', Syncroton_Command_FolderSync::FOLDERTYPE_DELETEDITEMS],
            ['Tasks', Syncroton_Command_FolderSync::FOLDERTYPE_TASK],
        ];

        $this->assertSame(strval(count($folders)), $xpath->query("//ns:FolderSync/ns:Changes/ns:Count")->item(0)->nodeValue);

        foreach ($folders as $idx => $folder) {
            $displayName = $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:DisplayName")->item($idx)->nodeValue;
            if (str_starts_with($folder[0], '/')) {
                $this->assertMatchesRegularExpression($folder[0], $displayName);
            } else {
                $this->assertSame($folder[0], $displayName);
            }

            $this->assertSame((string) $folder[1], $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:Type")->item($idx)->nodeValue);
            $this->assertSame('0', $xpath->query("//ns:FolderSync/ns:Changes/ns:Add/ns:ParentId")->item($idx)->nodeValue);
        }
    }
}
